// Fixed countdown to specific date
const targetDate = new Date();
targetDate.setDate(targetDate.getDate() + 7); // 7 days from now

let countdownInterval;
let customInterval;
let customPaused = false;
let customRemainingTime = 0;

// Elements
const daysEl = document.getElementById('days');
const hoursEl = document.getElementById('hours');
const minutesEl = document.getElementById('minutes');
const secondsEl = document.getElementById('seconds');
const progressEl = document.getElementById('progress');
const expiredMessage = document.getElementById('expired-message');
const resetBtn = document.getElementById('reset-btn');

// Custom timer elements
const customDaysInput = document.getElementById('custom-days');
const customHoursInput = document.getElementById('custom-hours');
const customMinutesInput = document.getElementById('custom-minutes');
const customSecondsInput = document.getElementById('custom-seconds');
const customDaysDisplay = document.getElementById('custom-days-display');
const customHoursDisplay = document.getElementById('custom-hours-display');
const customMinutesDisplay = document.getElementById('custom-minutes-display');
const customSecondsDisplay = document.getElementById('custom-seconds-display');
const startBtn = document.getElementById('start-btn');
const pauseBtn = document.getElementById('pause-btn');
const stopBtn = document.getElementById('stop-btn');

// Format number with leading zero
function formatTime(value) {
    return value < 10 ? `0${value}` : value;
}

// Calculate total duration for progress bar
const totalDuration = targetDate.getTime() - new Date().getTime();

// Update fixed countdown
function updateCountdown() {
    const now = new Date().getTime();
    const distance = targetDate.getTime() - now;

    if (distance < 0) {
        clearInterval(countdownInterval);
        daysEl.textContent = '00';
        hoursEl.textContent = '00';
        minutesEl.textContent = '00';
        secondsEl.textContent = '00';
        progressEl.style.width = '0%';
        expiredMessage.classList.add('active');
        return;
    }

    const days = Math.floor(distance / (1000 * 60 * 60 * 24));
    const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((distance % (1000 * 60)) / 1000);

    daysEl.textContent = formatTime(days);
    hoursEl.textContent = formatTime(hours);
    minutesEl.textContent = formatTime(minutes);
    secondsEl.textContent = formatTime(seconds);

    // Update progress bar
    const progressPercent = (distance / totalDuration) * 100;
    progressEl.style.width = `${progressPercent}%`;
}

// Reset fixed countdown
resetBtn.addEventListener('click', () => {
    expiredMessage.classList.remove('active');
    targetDate.setTime(new Date().getTime() + (7 * 24 * 60 * 60 * 1000));
    clearInterval(countdownInterval);
    updateCountdown();
    countdownInterval = setInterval(updateCountdown, 1000);
});

// Custom timer functions
function getCustomTimeInSeconds() {
    const days = parseInt(customDaysInput.value) || 0;
    const hours = parseInt(customHoursInput.value) || 0;
    const minutes = parseInt(customMinutesInput.value) || 0;
    const seconds = parseInt(customSecondsInput.value) || 0;
    return (days * 86400) + (hours * 3600) + (minutes * 60) + seconds;
}

function updateCustomDisplay(totalSeconds) {
    const days = Math.floor(totalSeconds / 86400);
    const hours = Math.floor((totalSeconds % 86400) / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = Math.floor(totalSeconds % 60);

    customDaysDisplay.textContent = formatTime(days);
    customHoursDisplay.textContent = formatTime(hours);
    customMinutesDisplay.textContent = formatTime(minutes);
    customSecondsDisplay.textContent = formatTime(seconds);
}

function startCustomTimer() {
    if (!customPaused) {
        customRemainingTime = getCustomTimeInSeconds();
        if (customRemainingTime <= 0) {
            alert('Veuillez définir un temps valide');
            return;
        }
    }

    startBtn.disabled = true;
    pauseBtn.disabled = false;
    stopBtn.disabled = false;
    customDaysInput.disabled = true;
    customHoursInput.disabled = true;
    customMinutesInput.disabled = true;
    customSecondsInput.disabled = true;

    customInterval = setInterval(() => {
        customRemainingTime--;
        updateCustomDisplay(customRemainingTime);

        if (customRemainingTime <= 0) {
            clearInterval(customInterval);
            playSound();
            alert('Temps écoulé !');
            resetCustomTimer();
        }
    }, 1000);

    customPaused = false;
}

function pauseCustomTimer() {
    clearInterval(customInterval);
    customPaused = true;
    startBtn.disabled = false;
    pauseBtn.disabled = true;
    startBtn.innerHTML = '<i class="fas fa-play"></i> Reprendre';
}

function stopCustomTimer() {
    clearInterval(customInterval);
    customPaused = false;
    resetCustomTimer();
}

function resetCustomTimer() {
    startBtn.disabled = false;
    pauseBtn.disabled = true;
    stopBtn.disabled = true;
    customDaysInput.disabled = false;
    customHoursInput.disabled = false;
    customMinutesInput.disabled = false;
    customSecondsInput.disabled = false;
    startBtn.innerHTML = '<i class="fas fa-play"></i> Démarrer';

    const initialTime = getCustomTimeInSeconds();
    updateCustomDisplay(initialTime);
}

function playSound() {
    // Create a simple beep sound using Web Audio API
    const audioContext = new (window.AudioContext || window.webkitAudioContext)();
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();

    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);

    oscillator.frequency.value = 800;
    oscillator.type = 'sine';

    gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);

    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.5);
}

// Event listeners
startBtn.addEventListener('click', startCustomTimer);
pauseBtn.addEventListener('click', pauseCustomTimer);
stopBtn.addEventListener('click', stopCustomTimer);

// Update custom display when inputs change
[customDaysInput, customHoursInput, customMinutesInput, customSecondsInput].forEach(input => {
    input.addEventListener('input', () => {
        if (!startBtn.disabled) {
            const totalSeconds = getCustomTimeInSeconds();
            updateCustomDisplay(totalSeconds);
        }
    });
});

// Initialize
updateCountdown();
countdownInterval = setInterval(updateCountdown, 1000);
updateCustomDisplay(getCustomTimeInSeconds());
