// Get all tooltip triggers
const tooltipTriggers = document.querySelectorAll('.tooltip-trigger');

// Function to create tooltip element
function createTooltip(text, position = 'top', color = null) {
    const tooltip = document.createElement('div');
    tooltip.classList.add('tooltip');
    tooltip.setAttribute('data-position', position);
    if (color) {
        tooltip.setAttribute('data-color', color);
    }
    tooltip.textContent = text;
    return tooltip;
}

// Function to show tooltip
function showTooltip(trigger) {
    const tooltipText = trigger.getAttribute('data-tooltip');
    const position = trigger.getAttribute('data-position') || 'top';
    const color = trigger.getAttribute('data-color');

    if (!tooltipText) return;

    // Create tooltip
    const tooltip = createTooltip(tooltipText, position, color);
    trigger.appendChild(tooltip);

    // Trigger reflow for animation
    setTimeout(() => {
        tooltip.classList.add('show');
    }, 10);

    // Store tooltip reference
    trigger._tooltip = tooltip;
}

// Function to hide tooltip
function hideTooltip(trigger) {
    const tooltip = trigger._tooltip;
    if (tooltip) {
        tooltip.classList.remove('show');
        setTimeout(() => {
            if (tooltip.parentNode) {
                tooltip.parentNode.removeChild(tooltip);
            }
        }, 300);
        trigger._tooltip = null;
    }
}

// Add event listeners to all tooltip triggers
tooltipTriggers.forEach(trigger => {
    // Mouse events
    trigger.addEventListener('mouseenter', () => {
        showTooltip(trigger);
    });

    trigger.addEventListener('mouseleave', () => {
        hideTooltip(trigger);
    });

    // Touch events for mobile
    trigger.addEventListener('touchstart', (e) => {
        // Prevent default to avoid triggering mouse events
        if (window.innerWidth <= 768) {
            showTooltip(trigger);

            // Hide tooltip after 2 seconds on mobile
            setTimeout(() => {
                hideTooltip(trigger);
            }, 2000);
        }
    });

    // Focus events for accessibility
    trigger.addEventListener('focus', () => {
        showTooltip(trigger);
    });

    trigger.addEventListener('blur', () => {
        hideTooltip(trigger);
    });
});

// Hide all tooltips when scrolling
let scrollTimeout;
window.addEventListener('scroll', () => {
    clearTimeout(scrollTimeout);

    // Hide all active tooltips
    tooltipTriggers.forEach(trigger => {
        if (trigger._tooltip) {
            hideTooltip(trigger);
        }
    });
}, { passive: true });

// Adjust tooltip position if it goes off-screen
function adjustTooltipPosition(tooltip, trigger) {
    const rect = tooltip.getBoundingClientRect();
    const triggerRect = trigger.getBoundingClientRect();

    // Check if tooltip goes off right edge
    if (rect.right > window.innerWidth) {
        tooltip.style.left = 'auto';
        tooltip.style.right = '0';
    }

    // Check if tooltip goes off left edge
    if (rect.left < 0) {
        tooltip.style.left = '0';
        tooltip.style.right = 'auto';
    }
}

// Optional: Update tooltip positions on window resize
window.addEventListener('resize', () => {
    tooltipTriggers.forEach(trigger => {
        if (trigger._tooltip) {
            hideTooltip(trigger);
        }
    });
});
