// Initialize carousels
const carousels = {};

// Setup carousel
function initCarousel(carouselId, itemCount) {
    carousels[carouselId] = {
        container: document.querySelector(`#${carouselId} .carousel-container`),
        currentRotation: 0,
        currentIndex: 0,
        itemCount: itemCount,
        angleStep: 360 / itemCount,
        isAutoplay: false,
        autoplayInterval: null
    };
}

// Rotate carousel
function rotateCarousel(carouselId, direction) {
    const carousel = carousels[carouselId];
    if (!carousel) return;

    carousel.currentRotation += direction * carousel.angleStep;
    carousel.currentIndex = (carousel.currentIndex - direction + carousel.itemCount) % carousel.itemCount;

    carousel.container.style.transform = `rotateY(${carousel.currentRotation}deg)`;

    // Update indicators if they exist
    updateIndicators(carouselId);
}

// Update indicators
function updateIndicators(carouselId) {
    const carouselElement = document.getElementById(carouselId);
    const indicators = carouselElement.querySelectorAll('.indicator');

    indicators.forEach((indicator, index) => {
        if (index === carousels[carouselId].currentIndex) {
            indicator.classList.add('active');
        } else {
            indicator.classList.remove('active');
        }
    });
}

// Go to specific slide
function goToSlide(carouselId, targetIndex) {
    const carousel = carousels[carouselId];
    if (!carousel) return;

    const currentIndex = carousel.currentIndex;
    const diff = targetIndex - currentIndex;

    carousel.currentRotation += diff * carousel.angleStep;
    carousel.currentIndex = targetIndex;

    carousel.container.style.transform = `rotateY(${carousel.currentRotation}deg)`;

    updateIndicators(carouselId);
}

// Autoplay functionality
function startAutoplay(carouselId, interval = 3000) {
    const carousel = carousels[carouselId];
    if (!carousel) return;

    carousel.isAutoplay = true;
    carousel.autoplayInterval = setInterval(() => {
        rotateCarousel(carouselId, -1);
    }, interval);
}

function stopAutoplay(carouselId) {
    const carousel = carousels[carouselId];
    if (!carousel) return;

    carousel.isAutoplay = false;
    if (carousel.autoplayInterval) {
        clearInterval(carousel.autoplayInterval);
        carousel.autoplayInterval = null;
    }
}

function toggleAutoplay(carouselId) {
    const carousel = carousels[carouselId];
    if (!carousel) return;

    if (carousel.isAutoplay) {
        stopAutoplay(carouselId);
    } else {
        startAutoplay(carouselId);
    }
}

// Initialize all carousels
document.addEventListener('DOMContentLoaded', () => {
    // Carousel 1 - 6 items
    initCarousel('carousel-1', 6);

    // Carousel 2 - 5 items (products)
    initCarousel('carousel-2', 5);

    // Carousel 3 - 8 items (autoplay)
    initCarousel('carousel-3', 8);
    startAutoplay('carousel-3', 3000);

    // Setup controls
    document.querySelectorAll('.prev-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const carouselId = btn.getAttribute('data-carousel');
            rotateCarousel(carouselId, 1);

            // Stop autoplay if clicking manually
            if (carousels[carouselId]?.isAutoplay) {
                stopAutoplay(carouselId);
            }
        });
    });

    document.querySelectorAll('.next-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const carouselId = btn.getAttribute('data-carousel');
            rotateCarousel(carouselId, -1);

            // Stop autoplay if clicking manually
            if (carousels[carouselId]?.isAutoplay) {
                stopAutoplay(carouselId);
            }
        });
    });

    // Setup pause/play button
    document.querySelectorAll('.pause-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const carouselId = btn.getAttribute('data-carousel');
            toggleAutoplay(carouselId);

            const icon = btn.querySelector('i');
            if (carousels[carouselId].isAutoplay) {
                icon.classList.remove('fa-play');
                icon.classList.add('fa-pause');
            } else {
                icon.classList.remove('fa-pause');
                icon.classList.add('fa-play');
            }
        });
    });

    // Setup indicators
    document.querySelectorAll('.indicator').forEach(indicator => {
        indicator.addEventListener('click', () => {
            const carouselElement = indicator.closest('.carousel-3d');
            const carouselId = carouselElement.id;
            const targetIndex = parseInt(indicator.getAttribute('data-index'));

            goToSlide(carouselId, targetIndex);

            // Stop autoplay if clicking manually
            if (carousels[carouselId]?.isAutoplay) {
                stopAutoplay(carouselId);
            }
        });
    });

    // Keyboard controls
    document.addEventListener('keydown', (e) => {
        if (e.key === 'ArrowLeft') {
            rotateCarousel('carousel-1', 1);
        } else if (e.key === 'ArrowRight') {
            rotateCarousel('carousel-1', -1);
        }
    });

    // Touch/Swipe support
    let touchStartX = 0;
    let touchEndX = 0;

    document.querySelectorAll('.carousel-3d').forEach(carousel => {
        carousel.addEventListener('touchstart', (e) => {
            touchStartX = e.changedTouches[0].screenX;
        });

        carousel.addEventListener('touchend', (e) => {
            touchEndX = e.changedTouches[0].screenX;
            handleSwipe(carousel.id);
        });
    });

    function handleSwipe(carouselId) {
        const swipeThreshold = 50;
        const diff = touchStartX - touchEndX;

        if (Math.abs(diff) > swipeThreshold) {
            if (diff > 0) {
                // Swipe left - go next
                rotateCarousel(carouselId, -1);
            } else {
                // Swipe right - go prev
                rotateCarousel(carouselId, 1);
            }

            // Stop autoplay if swiping
            if (carousels[carouselId]?.isAutoplay) {
                stopAutoplay(carouselId);
            }
        }
    }

    // Buy button handlers
    document.querySelectorAll('.buy-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.stopPropagation();

            const productCard = this.closest('.carousel-item');
            const productName = productCard.querySelector('h3').textContent;
            const price = productCard.querySelector('.price').textContent;

            // Visual feedback
            this.textContent = 'Ajouté !';
            this.style.background = '#10b981';

            setTimeout(() => {
                this.textContent = 'Acheter';
                this.style.background = '';
            }, 2000);

            console.log(`Produit ajouté: ${productName} - ${price}`);
        });
    });

    // Initialize indicators
    updateIndicators('carousel-1');
});

// Pause autoplay when page is not visible
document.addEventListener('visibilitychange', () => {
    if (document.hidden) {
        Object.keys(carousels).forEach(carouselId => {
            if (carousels[carouselId].isAutoplay) {
                stopAutoplay(carouselId);
                carousels[carouselId].wasAutoplayPaused = true;
            }
        });
    } else {
        Object.keys(carousels).forEach(carouselId => {
            if (carousels[carouselId].wasAutoplayPaused) {
                startAutoplay(carouselId);
                carousels[carouselId].wasAutoplayPaused = false;
            }
        });
    }
});
