// Initialize all dropzones
document.querySelectorAll('.dropzone').forEach(dropzone => {
    initDropzone(dropzone);
});

function initDropzone(dropzone) {
    const fileInput = dropzone.querySelector('.file-input');
    const previewContainer = dropzone.parentElement.querySelector('.files-preview');
    const maxFiles = parseInt(dropzone.dataset.maxFiles) || 1;
    const maxSize = 10 * 1024 * 1024; // 10 MB
    let uploadedFiles = [];

    // Click to select files
    dropzone.addEventListener('click', (e) => {
        if (e.target === dropzone || dropzone.contains(e.target)) {
            fileInput.click();
        }
    });

    // File input change
    fileInput.addEventListener('change', (e) => {
        handleFiles(Array.from(e.target.files));
    });

    // Drag and drop events
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        dropzone.addEventListener(eventName, preventDefaults);
    });

    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }

    // Drag over effect
    ['dragenter', 'dragover'].forEach(eventName => {
        dropzone.addEventListener(eventName, () => {
            dropzone.classList.add('drag-over');
        });
    });

    ['dragleave', 'drop'].forEach(eventName => {
        dropzone.addEventListener(eventName, () => {
            dropzone.classList.remove('drag-over');
        });
    });

    // Drop handler
    dropzone.addEventListener('drop', (e) => {
        const dt = e.dataTransfer;
        const files = Array.from(dt.files);
        handleFiles(files);
    });

    // Handle files
    function handleFiles(files) {
        // Clear previous error
        clearError();

        // Check max files
        const remainingSlots = maxFiles - uploadedFiles.length;
        if (files.length > remainingSlots) {
            showError(`Vous pouvez uploader seulement ${remainingSlots} fichier(s) supplémentaire(s)`);
            files = files.slice(0, remainingSlots);
        }

        if (files.length === 0) return;

        // Process each file
        files.forEach(file => {
            // Validate file size
            if (file.size > maxSize) {
                showError(`${file.name} est trop volumineux (max ${formatFileSize(maxSize)})`);
                return;
            }

            // Add to uploaded files
            uploadedFiles.push(file);

            // Create preview
            createFilePreview(file);

            // Simulate upload
            simulateUpload(file);
        });

        // Reset file input
        fileInput.value = '';
    }

    // Create file preview
    function createFilePreview(file) {
        const fileItem = document.createElement('div');
        fileItem.className = 'file-item';
        fileItem.dataset.fileName = file.name;

        // File preview or icon
        const isImage = file.type.startsWith('image/');
        let previewHTML = '';

        if (isImage) {
            const reader = new FileReader();
            reader.onload = (e) => {
                const img = fileItem.querySelector('.file-preview');
                if (img) {
                    img.src = e.target.result;
                }
            };
            reader.readAsDataURL(file);
            previewHTML = `<img class="file-preview" src="" alt="${escapeHtml(file.name)}">`;
        } else {
            const icon = getFileIcon(file.name);
            previewHTML = `
                <div class="file-icon-container">
                    <i class="${icon} file-icon"></i>
                </div>
            `;
        }

        fileItem.innerHTML = `
            ${previewHTML}
            <div class="file-info">
                <div class="file-name" title="${escapeHtml(file.name)}">${escapeHtml(file.name)}</div>
                <div class="file-size">${formatFileSize(file.size)}</div>
                <div class="file-progress">
                    <div class="progress-bar" style="width: 0%"></div>
                </div>
                <div class="file-status uploading">
                    <i class="fas fa-spinner fa-spin"></i>
                    <span>Upload en cours...</span>
                </div>
            </div>
            <button class="delete-btn" title="Supprimer">
                <i class="fas fa-times"></i>
            </button>
        `;

        // Delete button handler
        const deleteBtn = fileItem.querySelector('.delete-btn');
        deleteBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            removeFile(file.name);
        });

        previewContainer.appendChild(fileItem);
    }

    // Simulate file upload
    function simulateUpload(file) {
        const fileItem = previewContainer.querySelector(`[data-file-name="${file.name}"]`);
        const progressBar = fileItem.querySelector('.progress-bar');
        const statusElement = fileItem.querySelector('.file-status');

        let progress = 0;
        const interval = setInterval(() => {
            progress += Math.random() * 30;
            if (progress > 100) progress = 100;

            progressBar.style.width = progress + '%';

            if (progress >= 100) {
                clearInterval(interval);
                // Success state
                statusElement.className = 'file-status success';
                statusElement.innerHTML = `
                    <i class="fas fa-check-circle"></i>
                    <span>Upload réussi</span>
                `;
            }
        }, 200);
    }

    // Remove file
    function removeFile(fileName) {
        uploadedFiles = uploadedFiles.filter(f => f.name !== fileName);
        const fileItem = previewContainer.querySelector(`[data-file-name="${fileName}"]`);
        if (fileItem) {
            fileItem.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => fileItem.remove(), 300);
        }
    }

    // Show error
    function showError(message) {
        clearError();
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.innerHTML = `
            <i class="fas fa-exclamation-circle"></i>
            <p>${escapeHtml(message)}</p>
        `;
        dropzone.parentElement.insertBefore(errorDiv, previewContainer);

        // Auto-remove after 5 seconds
        setTimeout(() => errorDiv.remove(), 5000);
    }

    // Clear error
    function clearError() {
        const errorMessage = dropzone.parentElement.querySelector('.error-message');
        if (errorMessage) {
            errorMessage.remove();
        }
    }
}

// Get file icon based on extension
function getFileIcon(fileName) {
    const ext = fileName.split('.').pop().toLowerCase();
    const iconMap = {
        // Documents
        'pdf': 'fas fa-file-pdf',
        'doc': 'fas fa-file-word',
        'docx': 'fas fa-file-word',
        'txt': 'fas fa-file-alt',

        // Spreadsheets
        'xls': 'fas fa-file-excel',
        'xlsx': 'fas fa-file-excel',
        'csv': 'fas fa-file-excel',

        // Archives
        'zip': 'fas fa-file-archive',
        'rar': 'fas fa-file-archive',
        '7z': 'fas fa-file-archive',

        // Code
        'js': 'fas fa-file-code',
        'html': 'fas fa-file-code',
        'css': 'fas fa-file-code',
        'json': 'fas fa-file-code',

        // Images
        'jpg': 'fas fa-file-image',
        'jpeg': 'fas fa-file-image',
        'png': 'fas fa-file-image',
        'gif': 'fas fa-file-image',
        'svg': 'fas fa-file-image',

        // Video
        'mp4': 'fas fa-file-video',
        'avi': 'fas fa-file-video',
        'mov': 'fas fa-file-video',

        // Audio
        'mp3': 'fas fa-file-audio',
        'wav': 'fas fa-file-audio',
        'ogg': 'fas fa-file-audio'
    };

    return iconMap[ext] || 'fas fa-file';
}

// Format file size
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Slide out animation
const style = document.createElement('style');
style.textContent = `
    @keyframes slideOut {
        to {
            opacity: 0;
            transform: translateX(2rem);
        }
    }
`;
document.head.appendChild(style);
