// Elements
const codeInput = document.getElementById('codeInput');
const highlightedCode = document.getElementById('highlightedCode');
const lineNumbers = document.getElementById('lineNumbers');
const editorContainer = document.getElementById('editorContainer');
const languageSelect = document.getElementById('languageSelect');
const copyBtn = document.getElementById('copyBtn');
const clearBtn = document.getElementById('clearBtn');
const themeBtns = document.querySelectorAll('.theme-btn');
const currentLineSpan = document.getElementById('currentLine');
const charCountSpan = document.getElementById('charCount');
const exampleBtns = document.querySelectorAll('.example-btn');

// Current state
let currentLanguage = 'javascript';
let currentTheme = 'vscode';

// Code examples
const examples = {
    js: `// Fonction de tri rapide (QuickSort)
function quickSort(arr) {
    if (arr.length <= 1) return arr;

    const pivot = arr[arr.length - 1];
    const left = [];
    const right = [];

    for (let i = 0; i < arr.length - 1; i++) {
        if (arr[i] < pivot) {
            left.push(arr[i]);
        } else {
            right.push(arr[i]);
        }
    }

    return [...quickSort(left), pivot, ...quickSort(right)];
}

const numbers = [64, 34, 25, 12, 22, 11, 90];
console.log(quickSort(numbers));`,

    html: `<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ma Page Web</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <header class="header">
        <nav class="navigation">
            <ul>
                <li><a href="#home">Accueil</a></li>
                <li><a href="#about">À propos</a></li>
                <li><a href="#contact">Contact</a></li>
            </ul>
        </nav>
    </header>

    <main class="container">
        <h1>Bienvenue sur mon site</h1>
        <p>Ceci est un exemple de page HTML.</p>
    </main>

    <script src="script.js"></script>
</body>
</html>`,

    css: `.container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem;
}

.header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 1rem 0;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
}

.navigation ul {
    display: flex;
    list-style: none;
    gap: 2rem;
    justify-content: center;
}

.navigation a {
    color: white;
    text-decoration: none;
    font-weight: 500;
    transition: all 0.3s ease;
}

.navigation {
    transform: translateY(-2px);
    text-shadow: 0 2px 10px rgba(255, 255, 255, 0.5);
}

@media (max-width: 768px) {
    .navigation ul {
        flex-direction: column;
        gap: 1rem;
    }
}`,

    python: `# Classe pour gérer une liste de tâches
class TaskManager:
    def __init__(self):
        self.tasks = []

    def add_task(self, task, priority=1):
        """Ajoute une tâche avec une priorité"""
        self.tasks.append({
            'task': task,
            'priority': priority,
            'completed': False
        })

    def complete_task(self, index):
        """Marque une tâche comme complétée"""
        if 0 <= index < len(self.tasks):
            self.tasks[index]['completed'] = True

    def get_pending_tasks(self):
        """Retourne les tâches non complétées"""
        return [t for t in self.tasks if not t['completed']]

    def display_tasks(self):
        """Affiche toutes les tâches"""
        for i, task in enumerate(self.tasks):
            status = "✓" if task['completed'] else "○"
            print(f"{i+1}. [{status}] {task['task']} (Priorité: {task['priority']})")

# Utilisation
manager = TaskManager()
manager.add_task("Apprendre Python", 3)
manager.add_task("Créer un projet", 2)
manager.display_tasks()`
};

// Escape HTML
function escapeHtml(text) {
    return text
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}

// Highlight JavaScript
function highlightJS(code) {
    code = escapeHtml(code);

    // Temporary placeholders
    const placeholders = [];
    let placeholderIndex = 0;

    // Protect comments
    code = code.replace(/\/\/.*$/gm, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="comment">${match}</span>` });
        return placeholder;
    });

    code = code.replace(/\/\*[\s\S]*?\*\//g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="comment">${match}</span>` });
        return placeholder;
    });

    // Protect strings
    code = code.replace(/&quot;(?:[^&]|&(?!quot;))*&quot;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    code = code.replace(/&#39;(?:[^&]|&(?!#39;))*&#39;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    code = code.replace(/`(?:[^`\\]|\\.)*`/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    // Keywords
    code = code.replace(/\b(const|let|var|function|return|if|else|for|while|class|import|export|from|default|async|await|try|catch|throw|new|this|super|extends|static|get|set)\b/g, '<span class="keyword">$1</span>');

    // Builtins
    code = code.replace(/\b(console|document|window|Array|Object|String|Number|Boolean|Date|Math|JSON|Promise|Set|Map)\b/g, '<span class="builtin">$1</span>');

    // Functions
    code = code.replace(/\b([a-zA-Z_$][a-zA-Z0-9_$]*)\s*(?=\()/g, '<span class="function">$1</span>');

    // Numbers
    code = code.replace(/\b(\d+\.?\d*)\b/g, '<span class="number">$1</span>');

    // Restore placeholders
    placeholders.forEach(({ placeholder, replacement }) => {
        code = code.replace(placeholder, replacement);
    });

    return code;
}

// Highlight HTML
function highlightHTML(code) {
    code = escapeHtml(code);

    const placeholders = [];
    let placeholderIndex = 0;

    // Protect comments
    code = code.replace(/&lt;!--[\s\S]*?--&gt;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="comment">${match}</span>` });
        return placeholder;
    });

    // DOCTYPE
    code = code.replace(/&lt;!DOCTYPE[^&]*&gt;/gi, '<span class="keyword">$&</span>');

    // Tags with attributes
    code = code.replace(/&lt;(\/?)([\w-]+)((?:\s+[\w-]+=(?:&quot;[^&]*&quot;|&#39;[^&]*&#39;|[^\s&gt;]*))*)\s*(\/?&gt;)/g, (match, slash, tagName, attrs, closeBracket) => {
        let result = '&lt;' + slash;
        result += `<span class="tag">${tagName}</span>`;

        if (attrs) {
            result += attrs.replace(/([\w-]+)=(&quot;[^&]*&quot;|&#39;[^&]*&#39;)/g, (m, attrName, attrValue) => {
                return `<span class="attribute">${attrName}</span>=<span class="value">${attrValue}</span>`;
            });
        }

        result += closeBracket;
        return result;
    });

    // Restore placeholders
    placeholders.forEach(({ placeholder, replacement }) => {
        code = code.replace(placeholder, replacement);
    });

    return code;
}

// Highlight CSS
function highlightCSS(code) {
    code = escapeHtml(code);

    const placeholders = [];
    let placeholderIndex = 0;

    // Protect comments
    code = code.replace(/\/\*[\s\S]*?\*\//g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="comment">${match}</span>` });
        return placeholder;
    });

    // Protect strings
    code = code.replace(/&quot;[^&]*&quot;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    code = code.replace(/&#39;[^&]*&#39;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    // At-rules (@media, @keyframes, etc.)
    code = code.replace(/@(media|keyframes|import|charset|font-face|supports|page)\b/g, '<span class="keyword">@$1</span>');

    // Properties (before :) - do this FIRST to protect them
    code = code.replace(/(\s+)([\w-]+)(\s*):/g, (match, space1, prop, space2) => {
        return space1 + '<span class="property">' + prop + '</span>' + space2 + ':';
    });

    // Selectors - anything before { that's not a property
    code = code.replace(/(^|})(\s*)([\w\s.#:\-(),>+~[\]="']+?)(\s*)(\{)/gm, (match, before, space1, selector, space2, brace) => {
        const trimmed = selector.trim();
        // Skip if it looks like it's inside a block (has properties)
        if (trimmed && !trimmed.match(/<span class="property">/)) {
            return before + space1 + '<span class="selector">' + trimmed + '</span>' + space2 + brace;
        }
        return match;
    });

    // Numbers with units
    code = code.replace(/\b(\d+\.?\d*)(px|em|rem|%|vh|vw|vmin|vmax|ch|ex|s|ms|deg|grad|rad|turn)?\b/g, '<span class="number">$1$2</span>');

    // Restore placeholders
    placeholders.forEach(({ placeholder, replacement }) => {
        code = code.replace(placeholder, replacement);
    });

    return code;
}

// Highlight Python
function highlightPython(code) {
    code = escapeHtml(code);

    const placeholders = [];
    let placeholderIndex = 0;

    // Protect docstrings
    code = code.replace(/(&#39;&#39;&#39;[\s\S]*?&#39;&#39;&#39;|&quot;&quot;&quot;[\s\S]*?&quot;&quot;&quot;)/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    // Protect comments
    code = code.replace(/#.*$/gm, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="comment">${match}</span>` });
        return placeholder;
    });

    // Protect strings
    code = code.replace(/&quot;(?:[^&]|&(?!quot;))*&quot;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    code = code.replace(/&#39;(?:[^&]|&(?!#39;))*&#39;/g, (match) => {
        const placeholder = `___PLACEHOLDER_${placeholderIndex++}___`;
        placeholders.push({ placeholder, replacement: `<span class="string">${match}</span>` });
        return placeholder;
    });

    // Keywords
    code = code.replace(/\b(def|class|if|elif|else|for|while|return|import|from|as|try|except|finally|with|lambda|yield|pass|break|continue|raise|assert|global|nonlocal|and|or|not|in|is)\b/g, '<span class="keyword">$1</span>');

    // Special values
    code = code.replace(/\b(True|False|None)\b/g, '<span class="keyword">$1</span>');

    // Builtins
    code = code.replace(/\b(print|len|range|str|int|float|list|dict|set|tuple|open|input|type|isinstance|enumerate|zip|map|filter|sorted|sum|min|max|abs|round|self)\b/g, '<span class="builtin">$1</span>');

    // Functions
    code = code.replace(/\b([a-zA-Z_][a-zA-Z0-9_]*)\s*(?=\()/g, '<span class="function">$1</span>');

    // Numbers
    code = code.replace(/\b(\d+\.?\d*)\b/g, '<span class="number">$1</span>');

    // Restore placeholders
    placeholders.forEach(({ placeholder, replacement }) => {
        code = code.replace(placeholder, replacement);
    });

    return code;
}

// Main highlight function
function highlightCode(code, language) {
    if (language === 'javascript') {
        return highlightJS(code);
    } else if (language === 'html') {
        return highlightHTML(code);
    } else if (language === 'css') {
        return highlightCSS(code);
    } else if (language === 'python') {
        return highlightPython(code);
    }
    return escapeHtml(code);
}

// Update line numbers
function updateLineNumbers() {
    const lines = codeInput.value.split('\n').length;
    const lineNumbersHtml = Array.from({ length: lines }, (_, i) =>
        `<div class="line-number">${i + 1}</div>`
    ).join('');
    lineNumbers.innerHTML = lineNumbersHtml;
}

// Update status
function updateStatus() {
    const text = codeInput.value;
    const lines = text.split('\n');
    const cursorPosition = codeInput.selectionStart;
    const textBeforeCursor = text.substring(0, cursorPosition);
    const currentLine = textBeforeCursor.split('\n').length;

    currentLineSpan.textContent = currentLine;
    charCountSpan.textContent = text.length;
}

// Sync scroll
function syncScroll() {
    const highlight = document.getElementById('codeHighlight');
    highlight.scrollTop = codeInput.scrollTop;
    highlight.scrollLeft = codeInput.scrollLeft;
}

// Handle input
function handleInput() {
    const code = codeInput.value;
    highlightedCode.innerHTML = highlightCode(code, currentLanguage);
    updateLineNumbers();
    updateStatus();
}

// Handle keyboard shortcuts
function handleKeyDown(e) {
    // Tab key
    if (e.key === 'Tab') {
        e.preventDefault();
        const start = codeInput.selectionStart;
        const end = codeInput.selectionEnd;
        const value = codeInput.value;

        codeInput.value = value.substring(0, start) + '    ' + value.substring(end);
        codeInput.selectionStart = codeInput.selectionEnd = start + 4;
        handleInput();
    }

    // Ctrl+D - Duplicate line
    if ((e.ctrlKey || e.metaKey) && e.key === 'd') {
        e.preventDefault();
        const start = codeInput.selectionStart;
        const value = codeInput.value;
        const lines = value.split('\n');
        const textBeforeStart = value.substring(0, start);
        const currentLineIndex = textBeforeStart.split('\n').length - 1;
        const currentLine = lines[currentLineIndex];

        lines.splice(currentLineIndex + 1, 0, currentLine);
        codeInput.value = lines.join('\n');
        codeInput.selectionStart = codeInput.selectionEnd = start + currentLine.length + 1;
        handleInput();
    }

    // Ctrl+/ - Toggle comment
    if ((e.ctrlKey || e.metaKey) && e.key === '/') {
        e.preventDefault();
        const start = codeInput.selectionStart;
        const value = codeInput.value;
        const lines = value.split('\n');
        const textBeforeStart = value.substring(0, start);
        const currentLineIndex = textBeforeStart.split('\n').length - 1;
        const currentLine = lines[currentLineIndex];

        const commentChars = {
            javascript: '//',
            html: '<!--',
            css: '/*',
            python: '#'
        };

        const comment = commentChars[currentLanguage] || '//';

        if (currentLine.trim().startsWith(comment)) {
            lines[currentLineIndex] = currentLine.replace(new RegExp(`^(\\s*)${comment.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\s?`), '$1');
        } else {
            lines[currentLineIndex] = currentLine.replace(/^(\s*)/, `$1${comment} `);
        }

        codeInput.value = lines.join('\n');
        handleInput();
    }
}

// Copy code
function copyCode() {
    codeInput.select();
    document.execCommand('copy');
    showNotification('Code copié !');
}

// Clear code
function clearCode() {
    codeInput.value = '';
    handleInput();
    codeInput.focus();
}

// Change theme
function changeTheme(theme) {
    currentTheme = theme;
    editorContainer.setAttribute('data-theme', theme);
    themeBtns.forEach(btn => btn.classList.remove('active'));
    document.querySelector(`[data-theme="${theme}"]`).classList.add('active');
}

// Load example
function loadExample(type) {
    // Map example type to language
    const languageMap = {
        'js': 'javascript',
        'html': 'html',
        'css': 'css',
        'python': 'python'
    };

    const language = languageMap[type] || type;

    if (examples[type]) {
        codeInput.value = examples[type];
        currentLanguage = language;
        languageSelect.value = language;
        handleInput();
        codeInput.focus();
    }
}

// Show notification
function showNotification(message) {
    const notification = document.createElement('div');
    notification.className = 'copy-notification';
    notification.innerHTML = `<i class="fas fa-check-circle"></i> ${message}`;
    document.body.appendChild(notification);

    setTimeout(() => {
        notification.remove();
    }, 2000);
}

// Event listeners
codeInput.addEventListener('input', handleInput);
codeInput.addEventListener('scroll', syncScroll);
codeInput.addEventListener('keydown', handleKeyDown);
codeInput.addEventListener('click', updateStatus);
codeInput.addEventListener('keyup', updateStatus);

languageSelect.addEventListener('change', (e) => {
    currentLanguage = e.target.value;
    handleInput();
});

copyBtn.addEventListener('click', copyCode);
clearBtn.addEventListener('click', clearCode);

themeBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        changeTheme(btn.getAttribute('data-theme'));
    });
});

exampleBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        loadExample(btn.getAttribute('data-example'));
    });
});

// Initialize
editorContainer.setAttribute('data-theme', currentTheme);
updateLineNumbers();
updateStatus();
codeInput.focus();
