// Elements
const comparisonSlider = document.getElementById('comparisonSlider');
const sliderHandle = document.getElementById('sliderHandle');
const imageAfter = document.getElementById('imageAfter');

// State
let isDragging = false;
let sliderRect = null;

// Initialize slider position at 50%
let sliderPosition = 50;

// Update slider position
function updateSliderPosition(percentage) {
    // Clamp between 0 and 100
    percentage = Math.max(0, Math.min(100, percentage));
    sliderPosition = percentage;

    // Update handle position
    sliderHandle.style.left = `${percentage}%`;

    // Update image-after clip-path
    // clip-path: inset(top right bottom left)
    // We want to clip from the left side based on percentage
    const clipPercentage = 100 - percentage;
    imageAfter.style.clipPath = `inset(0 0 0 ${percentage}%)`;
}

// Get position from mouse/touch event
function getPositionFromEvent(event) {
    if (!sliderRect) {
        sliderRect = comparisonSlider.getBoundingClientRect();
    }

    const clientX = event.type.includes('touch')
        ? event.touches[0].clientX
        : event.clientX;

    const x = clientX - sliderRect.left;
    const percentage = (x / sliderRect.width) * 100;

    return percentage;
}

// Start dragging
function startDrag(event) {
    event.preventDefault();
    isDragging = true;
    sliderRect = comparisonSlider.getBoundingClientRect();
    comparisonSlider.classList.add('dragging');

    // Update position immediately
    const percentage = getPositionFromEvent(event);
    updateSliderPosition(percentage);
}

// While dragging
function onDrag(event) {
    if (!isDragging) return;

    event.preventDefault();
    const percentage = getPositionFromEvent(event);
    updateSliderPosition(percentage);
}

// Stop dragging
function stopDrag() {
    if (!isDragging) return;
    isDragging = false;
    sliderRect = null;
    comparisonSlider.classList.remove('dragging');
}

// Mouse events
comparisonSlider.addEventListener('mousedown', startDrag);
document.addEventListener('mousemove', onDrag);
document.addEventListener('mouseup', stopDrag);

// Touch events
comparisonSlider.addEventListener('touchstart', startDrag, { passive: false });
document.addEventListener('touchmove', onDrag, { passive: false });
document.addEventListener('touchend', stopDrag);

// Update slider rect on window resize
window.addEventListener('resize', () => {
    if (!isDragging) {
        sliderRect = null;
        // Reapply current position
        updateSliderPosition(sliderPosition);
    }
});

// Keyboard accessibility (optional enhancement)
sliderHandle.setAttribute('tabindex', '0');
sliderHandle.addEventListener('keydown', (event) => {
    let newPosition = sliderPosition;

    switch(event.key) {
        case 'ArrowLeft':
            event.preventDefault();
            newPosition = Math.max(0, sliderPosition - 2);
            break;
        case 'ArrowRight':
            event.preventDefault();
            newPosition = Math.min(100, sliderPosition + 2);
            break;
        case 'Home':
            event.preventDefault();
            newPosition = 0;
            break;
        case 'End':
            event.preventDefault();
            newPosition = 100;
            break;
    }

    if (newPosition !== sliderPosition) {
        updateSliderPosition(newPosition);
    }
});

// Initialize
updateSliderPosition(50);
