// Toast notification system
class ToastManager {
    constructor() {
        this.container = document.getElementById('toastContainer');
        this.toasts = [];
        this.idCounter = 0;
    }

    /**
     * Show a toast notification
     * @param {Object} options - Toast configuration
     * @param {string} options.type - Type: 'success', 'error', 'warning', 'info'
     * @param {string} options.title - Toast title
     * @param {string} options.message - Toast message
     * @param {number} options.duration - Duration in ms (default: 5000)
     * @param {Array} options.actions - Optional action buttons
     */
    show(options) {
        const {
            type = 'info',
            title = '',
            message = '',
            duration = 5000,
            actions = []
        } = options;

        const id = this.idCounter++;
        const toast = this.createToast(id, type, title, message, duration, actions);

        this.container.appendChild(toast);
        this.toasts.push({ id, element: toast, duration });

        // Auto-dismiss after duration
        if (duration > 0) {
            setTimeout(() => {
                this.remove(id);
            }, duration);
        }

        return id;
    }

    createToast(id, type, title, message, duration, actions) {
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.dataset.id = id;

        // Icon based on type
        const icons = {
            success: 'fa-check-circle',
            error: 'fa-times-circle',
            warning: 'fa-exclamation-triangle',
            info: 'fa-info-circle'
        };

        const icon = icons[type] || icons.info;

        toast.innerHTML = `
            <div class="toast-icon">
                <i class="fas ${icon}"></i>
            </div>
            <div class="toast-content">
                ${title ? `<div class="toast-title">${title}</div>` : ''}
                ${message ? `<div class="toast-message">${message}</div>` : ''}
                ${actions.length > 0 ? this.createActions(actions, id) : ''}
            </div>
            <button class="toast-close" onclick="toastManager.remove(${id})">
                <i class="fas fa-times"></i>
            </button>
            ${duration > 0 ? `<div class="toast-progress" style="animation-duration: ${duration}ms"></div>` : ''}
        `;

        return toast;
    }

    createActions(actions, toastId) {
        const actionsHtml = actions.map(action => {
            const onClick = action.onClick ?
                `toastManager.handleAction(${toastId}, '${action.label}', ${action.onClick})` :
                `toastManager.remove(${toastId})`;

            return `<button class="toast-action" onclick="${onClick}">${action.label}</button>`;
        }).join('');

        return `<div class="toast-actions">${actionsHtml}</div>`;
    }

    handleAction(toastId, label, callback) {
        if (callback && typeof callback === 'function') {
            callback();
        }
        this.remove(toastId);
    }

    remove(id) {
        const toastData = this.toasts.find(t => t.id === id);
        if (!toastData) return;

        const toast = toastData.element;
        toast.classList.add('removing');

        setTimeout(() => {
            toast.remove();
            this.toasts = this.toasts.filter(t => t.id !== id);
        }, 300);
    }

    removeAll() {
        this.toasts.forEach(t => this.remove(t.id));
    }

    // Shorthand methods
    success(title, message, options = {}) {
        return this.show({ type: 'success', title, message, ...options });
    }

    error(title, message, options = {}) {
        return this.show({ type: 'error', title, message, ...options });
    }

    warning(title, message, options = {}) {
        return this.show({ type: 'warning', title, message, ...options });
    }

    info(title, message, options = {}) {
        return this.show({ type: 'info', title, message, ...options });
    }
}

// Initialize toast manager
const toastManager = new ToastManager();

// Demo functions
function showSuccessToast() {
    toastManager.success(
        'Opération réussie !',
        'Vos modifications ont été enregistrées avec succès.'
    );
}

function showErrorToast() {
    toastManager.error(
        'Erreur critique',
        'Impossible de se connecter au serveur. Veuillez réessayer.',
        { duration: 7000 }
    );
}

function showWarningToast() {
    toastManager.warning(
        'Attention !',
        'Vous avez des modifications non sauvegardées.'
    );
}

function showInfoToast() {
    toastManager.info(
        'Nouvelle mise à jour',
        'Une nouvelle version est disponible (v2.5.0)'
    );
}

function showActionToast() {
    toastManager.show({
        type: 'info',
        title: 'Fichier supprimé',
        message: 'Document.pdf a été supprimé',
        duration: 8000,
        actions: [
            {
                label: 'Annuler',
                onClick: function() {
                    toastManager.success('Annulé', 'Le fichier a été restauré');
                }
            }
        ]
    });
}

function showMultipleToasts() {
    setTimeout(() => toastManager.success('Toast 1', 'Premier message'), 0);
    setTimeout(() => toastManager.info('Toast 2', 'Deuxième message'), 200);
    setTimeout(() => toastManager.warning('Toast 3', 'Troisième message'), 400);
    setTimeout(() => toastManager.error('Toast 4', 'Quatrième message'), 600);
}

// Position change
function changePosition(position) {
    toastManager.container.className = `toast-container ${position}`;

    // Update active button
    document.querySelectorAll('.position-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-position="${position}"]`).classList.add('active');
}

// Example: programmatic usage
// toastManager.success('Title', 'Message');
// toastManager.error('Title', 'Message', { duration: 10000 });
// toastManager.show({ type: 'warning', title: 'Title', message: 'Message', actions: [...] });
