// Range Slider Class
class RangeSlider {
    constructor(element, options = {}) {
        this.element = element;
        this.min = parseFloat(element.dataset.min) || options.min || 0;
        this.max = parseFloat(element.dataset.max) || options.max || 100;
        this.step = parseFloat(element.dataset.step) || options.step || 1;
        this.prefix = element.dataset.prefix || options.prefix || '';
        this.suffix = element.dataset.suffix || options.suffix || '';

        this.valueMin = options.valueMin || this.min;
        this.valueMax = options.valueMax || this.max;

        this.minInput = options.minInput;
        this.maxInput = options.maxInput;
        this.minDisplay = options.minDisplay;
        this.maxDisplay = options.maxDisplay;

        this.isDragging = false;
        this.activeHandle = null;

        this.init();
    }

    init() {
        this.createSlider();
        this.attachEvents();
        this.update();
    }

    createSlider() {
        // Create slider structure
        this.element.innerHTML = `
            <div class="slider-track"></div>
            <div class="slider-range"></div>
            <div class="slider-handle" data-handle="min"></div>
            <div class="slider-handle" data-handle="max"></div>
        `;

        this.track = this.element.querySelector('.slider-track');
        this.range = this.element.querySelector('.slider-range');
        this.handleMin = this.element.querySelector('[data-handle="min"]');
        this.handleMax = this.element.querySelector('[data-handle="max"]');
    }

    attachEvents() {
        // Handle dragging
        this.handleMin.addEventListener('mousedown', (e) => this.startDrag(e, 'min'));
        this.handleMax.addEventListener('mousedown', (e) => this.startDrag(e, 'max'));

        this.handleMin.addEventListener('touchstart', (e) => this.startDrag(e, 'min'), { passive: false });
        this.handleMax.addEventListener('touchstart', (e) => this.startDrag(e, 'max'), { passive: false });

        document.addEventListener('mousemove', (e) => this.onDrag(e));
        document.addEventListener('mouseup', () => this.stopDrag());

        document.addEventListener('touchmove', (e) => this.onDrag(e), { passive: false });
        document.addEventListener('touchend', () => this.stopDrag());

        // Click on track to move nearest handle
        this.track.addEventListener('click', (e) => this.onTrackClick(e));

        // Input synchronization
        if (this.minInput) {
            this.minInput.addEventListener('input', () => {
                this.valueMin = this.clamp(parseFloat(this.minInput.value) || this.min, this.min, this.valueMax);
                this.update();
            });
        }

        if (this.maxInput) {
            this.maxInput.addEventListener('input', () => {
                this.valueMax = this.clamp(parseFloat(this.maxInput.value) || this.max, this.valueMin, this.max);
                this.update();
            });
        }
    }

    startDrag(e, handle) {
        e.preventDefault();
        this.isDragging = true;
        this.activeHandle = handle;

        if (handle === 'min') {
            this.handleMin.classList.add('dragging');
        } else {
            this.handleMax.classList.add('dragging');
        }
    }

    onDrag(e) {
        if (!this.isDragging) return;

        e.preventDefault();

        const clientX = e.type.includes('touch') ? e.touches[0].clientX : e.clientX;
        const rect = this.track.getBoundingClientRect();
        const x = clientX - rect.left;
        const percentage = this.clamp(x / rect.width, 0, 1);
        const value = this.snapToStep(this.min + percentage * (this.max - this.min));

        if (this.activeHandle === 'min') {
            this.valueMin = this.clamp(value, this.min, this.valueMax);
        } else {
            this.valueMax = this.clamp(value, this.valueMin, this.max);
        }

        this.update();
    }

    stopDrag() {
        if (!this.isDragging) return;

        this.isDragging = false;
        this.handleMin.classList.remove('dragging');
        this.handleMax.classList.remove('dragging');
        this.activeHandle = null;
    }

    onTrackClick(e) {
        if (this.isDragging) return;

        const rect = this.track.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const percentage = x / rect.width;
        const value = this.snapToStep(this.min + percentage * (this.max - this.min));

        // Move the nearest handle
        const distToMin = Math.abs(value - this.valueMin);
        const distToMax = Math.abs(value - this.valueMax);

        if (distToMin < distToMax) {
            this.valueMin = this.clamp(value, this.min, this.valueMax);
        } else {
            this.valueMax = this.clamp(value, this.valueMin, this.max);
        }

        this.update();
    }

    update() {
        // Update handle positions
        const minPercent = ((this.valueMin - this.min) / (this.max - this.min)) * 100;
        const maxPercent = ((this.valueMax - this.min) / (this.max - this.min)) * 100;

        this.handleMin.style.left = `${minPercent}%`;
        this.handleMax.style.left = `${maxPercent}%`;

        // Update range bar
        this.range.style.left = `${minPercent}%`;
        this.range.style.width = `${maxPercent - minPercent}%`;

        // Update inputs
        if (this.minInput) {
            this.minInput.value = this.valueMin;
        }
        if (this.maxInput) {
            this.maxInput.value = this.valueMax;
        }

        // Update displays
        if (this.minDisplay) {
            this.minDisplay.textContent = this.formatValue(this.valueMin);
        }
        if (this.maxDisplay) {
            this.maxDisplay.textContent = this.formatValue(this.valueMax);
        }
    }

    formatValue(value) {
        return `${this.prefix}${value}${this.suffix}`;
    }

    snapToStep(value) {
        return Math.round(value / this.step) * this.step;
    }

    clamp(value, min, max) {
        return Math.min(Math.max(value, min), max);
    }

    setValues(min, max) {
        this.valueMin = this.clamp(min, this.min, this.max);
        this.valueMax = this.clamp(max, this.min, this.max);

        // Ensure min <= max
        if (this.valueMin > this.valueMax) {
            [this.valueMin, this.valueMax] = [this.valueMax, this.valueMin];
        }

        this.update();
    }
}

// Initialize sliders
let priceSlider, ageSlider, percentSlider;

document.addEventListener('DOMContentLoaded', () => {
    // Price Slider
    priceSlider = new RangeSlider(document.getElementById('priceSlider'), {
        valueMin: 0,
        valueMax: 1000,
        minInput: document.getElementById('priceMinInput'),
        maxInput: document.getElementById('priceMaxInput'),
        minDisplay: document.getElementById('priceMinValue'),
        maxDisplay: document.getElementById('priceMaxValue')
    });

    // Age Slider
    ageSlider = new RangeSlider(document.getElementById('ageSlider'), {
        valueMin: 18,
        valueMax: 65,
        minInput: document.getElementById('ageMinInput'),
        maxInput: document.getElementById('ageMaxInput'),
        minDisplay: document.getElementById('ageMinValue'),
        maxDisplay: document.getElementById('ageMaxValue')
    });

    // Percentage Slider
    percentSlider = new RangeSlider(document.getElementById('percentSlider'), {
        valueMin: 0,
        valueMax: 100,
        minInput: document.getElementById('percentMinInput'),
        maxInput: document.getElementById('percentMaxInput'),
        minDisplay: document.getElementById('percentMinValue'),
        maxDisplay: document.getElementById('percentMaxValue')
    });
});

// Preset functions
function setPriceRange(min, max) {
    priceSlider.setValues(min, max);
}
