// Holographic Card Effect
class HolographicCard {
    constructor(element) {
        this.card = element;
        this.holoLayer = this.card.querySelector('.holo-layer');
        this.shine = this.card.querySelector('.shine');

        this.bounds = null;
        this.isHovering = false;

        this.init();
    }

    init() {
        // Mouse events
        this.card.addEventListener('mouseenter', () => this.handleMouseEnter());
        this.card.addEventListener('mousemove', (e) => this.handleMouseMove(e));
        this.card.addEventListener('mouseleave', () => this.handleMouseLeave());

        // Touch events for mobile
        this.card.addEventListener('touchstart', (e) => this.handleTouchStart(e));
        this.card.addEventListener('touchmove', (e) => this.handleTouchMove(e));
        this.card.addEventListener('touchend', () => this.handleTouchEnd());
    }

    handleMouseEnter() {
        this.isHovering = true;
        this.bounds = this.card.getBoundingClientRect();
    }

    handleMouseMove(e) {
        if (!this.isHovering) return;

        // Calculate mouse position relative to card
        const mouseX = e.clientX - this.bounds.left;
        const mouseY = e.clientY - this.bounds.top;

        // Convert to percentage
        const xPercent = (mouseX / this.bounds.width) * 100;
        const yPercent = (mouseY / this.bounds.height) * 100;

        // Calculate rotation angles (inverted for realistic effect)
        const rotateX = ((yPercent - 50) / 50) * -10; // Max ±10deg
        const rotateY = ((xPercent - 50) / 50) * 10;  // Max ±10deg

        // Apply 3D transform
        this.card.style.transform = `
            perspective(1000px)
            rotateX(${rotateX}deg)
            rotateY(${rotateY}deg)
            scale3d(1.02, 1.02, 1.02)
        `;

        // Update holographic layer position
        this.holoLayer.style.setProperty('--pointer-x', `${xPercent}%`);
        this.holoLayer.style.setProperty('--pointer-y', `${yPercent}%`);

        // Update shine position
        this.shine.style.setProperty('--pointer-x', `${xPercent}%`);
        this.shine.style.setProperty('--pointer-y', `${yPercent}%`);

        // Add dynamic background position to shine for movement effect
        const shineX = (xPercent - 50) * 2;
        const shineY = (yPercent - 50) * 2;
        this.shine.style.backgroundPosition = `${50 + shineX}% ${50 + shineY}%`;
    }

    handleMouseLeave() {
        this.isHovering = false;

        // Reset transform with smooth transition
        this.card.style.transform = 'perspective(1000px) rotateX(0deg) rotateY(0deg) scale3d(1, 1, 1)';

        // Reset holographic layer
        this.holoLayer.style.setProperty('--pointer-x', '50%');
        this.holoLayer.style.setProperty('--pointer-y', '50%');

        // Reset shine
        this.shine.style.backgroundPosition = '50% 50%';
    }

    handleTouchStart(e) {
        e.preventDefault();
        this.isHovering = true;
        this.bounds = this.card.getBoundingClientRect();

        // Trigger hover effects
        this.holoLayer.style.opacity = '0.5';
        this.shine.style.opacity = '1';
    }

    handleTouchMove(e) {
        if (!this.isHovering) return;
        e.preventDefault();

        const touch = e.touches[0];
        const mouseX = touch.clientX - this.bounds.left;
        const mouseY = touch.clientY - this.bounds.top;

        const xPercent = (mouseX / this.bounds.width) * 100;
        const yPercent = (mouseY / this.bounds.height) * 100;

        const rotateX = ((yPercent - 50) / 50) * -10;
        const rotateY = ((xPercent - 50) / 50) * 10;

        this.card.style.transform = `
            perspective(1000px)
            rotateX(${rotateX}deg)
            rotateY(${rotateY}deg)
            scale3d(1.02, 1.02, 1.02)
        `;

        this.holoLayer.style.setProperty('--pointer-x', `${xPercent}%`);
        this.holoLayer.style.setProperty('--pointer-y', `${yPercent}%`);

        this.shine.style.setProperty('--pointer-x', `${xPercent}%`);
        this.shine.style.setProperty('--pointer-y', `${yPercent}%`);

        const shineX = (xPercent - 50) * 2;
        const shineY = (yPercent - 50) * 2;
        this.shine.style.backgroundPosition = `${50 + shineX}% ${50 + shineY}%`;
    }

    handleTouchEnd() {
        this.isHovering = false;

        this.card.style.transform = 'perspective(1000px) rotateX(0deg) rotateY(0deg) scale3d(1, 1, 1)';

        this.holoLayer.style.opacity = '0';
        this.shine.style.opacity = '0';

        this.holoLayer.style.setProperty('--pointer-x', '50%');
        this.holoLayer.style.setProperty('--pointer-y', '50%');

        this.shine.style.backgroundPosition = '50% 50%';
    }
}

// Initialize all holographic cards
document.addEventListener('DOMContentLoaded', () => {
    const cards = document.querySelectorAll('.holo-card');
    cards.forEach(card => new HolographicCard(card));
});

// Add subtle breathing animation when idle
document.addEventListener('DOMContentLoaded', () => {
    const cards = document.querySelectorAll('.holo-card');

    cards.forEach((card, index) => {
        // Stagger the animation
        const delay = index * 0.2;
        card.style.animation = `cardBreathe 4s ease-in-out ${delay}s infinite`;
    });
});

// Add breathing animation keyframes via JavaScript
const style = document.createElement('style');
style.textContent = `
    @keyframes cardBreathe {
        0%, 100% {
            box-shadow:
                0 20px 60px rgba(0, 0, 0, 0.5),
                0 0 40px rgba(102, 126, 234, 0.1);
        }
        50% {
            box-shadow:
                0 25px 70px rgba(0, 0, 0, 0.6),
                0 0 60px rgba(102, 126, 234, 0.2);
        }
    }
`;
document.head.appendChild(style);
