// Command database
const commands = [
    // Navigation
    {
        category: 'Navigation',
        name: 'Aller à l\'accueil',
        description: 'Retourner à la page d\'accueil',
        icon: 'fas fa-home',
        shortcut: ['G', 'H'],
        action: () => console.log('Navigation vers accueil')
    },
    {
        category: 'Navigation',
        name: 'Aller au tableau de bord',
        description: 'Ouvrir le tableau de bord',
        icon: 'fas fa-tachometer-alt',
        shortcut: ['G', 'D'],
        action: () => console.log('Navigation vers dashboard')
    },
    {
        category: 'Navigation',
        name: 'Aller aux paramètres',
        description: 'Ouvrir les paramètres',
        icon: 'fas fa-cog',
        shortcut: ['G', 'S'],
        action: () => console.log('Navigation vers paramètres')
    },
    {
        category: 'Navigation',
        name: 'Aller au profil',
        description: 'Voir votre profil utilisateur',
        icon: 'fas fa-user',
        shortcut: ['G', 'P'],
        action: () => console.log('Navigation vers profil')
    },
    {
        category: 'Navigation',
        name: 'Aller aux notifications',
        description: 'Voir vos notifications',
        icon: 'fas fa-bell',
        shortcut: ['G', 'N'],
        action: () => console.log('Navigation vers notifications')
    },

    // Actions
    {
        category: 'Actions',
        name: 'Créer un nouveau document',
        description: 'Créer un nouveau document vide',
        icon: 'fas fa-file-alt',
        shortcut: ['Ctrl', 'N'],
        action: () => console.log('Nouveau document créé')
    },
    {
        category: 'Actions',
        name: 'Sauvegarder',
        description: 'Sauvegarder les modifications',
        icon: 'fas fa-save',
        shortcut: ['Ctrl', 'S'],
        action: () => console.log('Sauvegarde effectuée')
    },
    {
        category: 'Actions',
        name: 'Rechercher',
        description: 'Rechercher dans le contenu',
        icon: 'fas fa-search',
        shortcut: ['Ctrl', 'F'],
        action: () => console.log('Recherche lancée')
    },
    {
        category: 'Actions',
        name: 'Imprimer',
        description: 'Imprimer le document actuel',
        icon: 'fas fa-print',
        shortcut: ['Ctrl', 'P'],
        action: () => console.log('Impression lancée')
    },
    {
        category: 'Actions',
        name: 'Copier',
        description: 'Copier la sélection',
        icon: 'fas fa-copy',
        shortcut: ['Ctrl', 'C'],
        action: () => console.log('Copie effectuée')
    },
    {
        category: 'Actions',
        name: 'Coller',
        description: 'Coller depuis le presse-papier',
        icon: 'fas fa-paste',
        shortcut: ['Ctrl', 'V'],
        action: () => console.log('Collage effectué')
    },

    // Apparence
    {
        category: 'Apparence',
        name: 'Basculer le mode sombre',
        description: 'Activer/désactiver le thème sombre',
        icon: 'fas fa-moon',
        shortcut: ['Ctrl', 'D'],
        action: () => console.log('Mode sombre basculé')
    },
    {
        category: 'Apparence',
        name: 'Augmenter le zoom',
        description: 'Agrandir l\'affichage',
        icon: 'fas fa-search-plus',
        shortcut: ['Ctrl', '+'],
        action: () => console.log('Zoom augmenté')
    },
    {
        category: 'Apparence',
        name: 'Diminuer le zoom',
        description: 'Réduire l\'affichage',
        icon: 'fas fa-search-minus',
        shortcut: ['Ctrl', '-'],
        action: () => console.log('Zoom diminué')
    },
    {
        category: 'Apparence',
        name: 'Réinitialiser le zoom',
        description: 'Revenir au zoom par défaut',
        icon: 'fas fa-compress',
        shortcut: ['Ctrl', '0'],
        action: () => console.log('Zoom réinitialisé')
    },

    // Compte
    {
        category: 'Compte',
        name: 'Se déconnecter',
        description: 'Fermer votre session',
        icon: 'fas fa-sign-out-alt',
        shortcut: [],
        action: () => console.log('Déconnexion')
    },
    {
        category: 'Compte',
        name: 'Modifier le profil',
        description: 'Éditer vos informations personnelles',
        icon: 'fas fa-user-edit',
        shortcut: [],
        action: () => console.log('Édition du profil')
    },
    {
        category: 'Compte',
        name: 'Changer le mot de passe',
        description: 'Modifier votre mot de passe',
        icon: 'fas fa-key',
        shortcut: [],
        action: () => console.log('Changement de mot de passe')
    },

    // Aide
    {
        category: 'Aide',
        name: 'Documentation',
        description: 'Consulter la documentation',
        icon: 'fas fa-book',
        shortcut: ['?'],
        action: () => console.log('Documentation ouverte')
    },
    {
        category: 'Aide',
        name: 'Raccourcis clavier',
        description: 'Afficher tous les raccourcis',
        icon: 'fas fa-keyboard',
        shortcut: [],
        action: () => console.log('Raccourcis affichés')
    },
    {
        category: 'Aide',
        name: 'Contacter le support',
        description: 'Obtenir de l\'aide',
        icon: 'fas fa-question-circle',
        shortcut: [],
        action: () => console.log('Support contacté')
    }
];

// DOM Elements
const overlay = document.getElementById('commandPalette');
const searchInput = document.getElementById('commandSearch');
const commandList = document.getElementById('commandList');
const emptyState = document.getElementById('emptyState');

// State
let selectedIndex = 0;
let filteredCommands = [];

// Fuzzy search with scoring
function fuzzyMatchWithScore(query, text) {
    if (!query) return { matched: true, score: 0, highlights: [] };

    query = query.toLowerCase();
    text = text.toLowerCase();

    let score = 0;
    let queryIndex = 0;
    let consecutiveMatches = 0;
    const highlights = [];

    for (let i = 0; i < text.length && queryIndex < query.length; i++) {
        if (text[i] === query[queryIndex]) {
            // Track highlight positions
            highlights.push(i);

            // Calculate score
            score += 1 + consecutiveMatches;
            consecutiveMatches++;
            queryIndex++;

            // Bonus for start of word
            if (i === 0 || text[i - 1] === ' ') {
                score += 5;
            }
        } else {
            consecutiveMatches = 0;
        }
    }

    // If not all characters matched, return no match
    if (queryIndex !== query.length) {
        return { matched: false, score: 0, highlights: [] };
    }

    // Bonus for exact match
    if (text === query) {
        score += 20;
    }

    // Bonus for match at start
    if (text.startsWith(query)) {
        score += 10;
    }

    return { matched: true, score, highlights };
}

// Highlight matching characters
function highlightMatch(text, highlights) {
    if (!highlights || highlights.length === 0) {
        return escapeHtml(text);
    }

    let result = '';
    for (let i = 0; i < text.length; i++) {
        if (highlights.includes(i)) {
            result += `<mark>${escapeHtml(text[i])}</mark>`;
        } else {
            result += escapeHtml(text[i]);
        }
    }
    return result;
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Filter and render commands
function filterCommands(query) {
    // Filter commands with fuzzy search
    const results = commands
        .map(cmd => {
            const nameMatch = fuzzyMatchWithScore(query, cmd.name);
            const descMatch = fuzzyMatchWithScore(query, cmd.description);

            // Use the best match
            const bestMatch = nameMatch.score > descMatch.score ? nameMatch : descMatch;

            return {
                ...cmd,
                matched: nameMatch.matched || descMatch.matched,
                score: Math.max(nameMatch.score, descMatch.score),
                nameHighlights: nameMatch.highlights,
                descHighlights: descMatch.highlights
            };
        })
        .filter(cmd => cmd.matched)
        .sort((a, b) => b.score - a.score);

    filteredCommands = results;
    renderCommands();
}

// Render commands
function renderCommands() {
    commandList.innerHTML = '';

    if (filteredCommands.length === 0) {
        commandList.style.display = 'none';
        emptyState.style.display = 'block';
        return;
    }

    commandList.style.display = 'block';
    emptyState.style.display = 'none';

    // Group by category
    const categories = {};
    filteredCommands.forEach(cmd => {
        if (!categories[cmd.category]) {
            categories[cmd.category] = [];
        }
        categories[cmd.category].push(cmd);
    });

    // Render each category
    Object.keys(categories).forEach(category => {
        const categoryDiv = document.createElement('div');
        categoryDiv.className = 'command-category';

        const label = document.createElement('div');
        label.className = 'category-label';
        label.textContent = category;
        categoryDiv.appendChild(label);

        categories[category].forEach((cmd, index) => {
            const globalIndex = filteredCommands.indexOf(cmd);
            const item = document.createElement('div');
            item.className = 'command-item';
            if (globalIndex === selectedIndex) {
                item.classList.add('selected');
            }

            item.innerHTML = `
                <i class="${cmd.icon} command-icon"></i>
                <div class="command-info">
                    <div class="command-name">${highlightMatch(cmd.name, cmd.nameHighlights)}</div>
                    <div class="command-description">${escapeHtml(cmd.description)}</div>
                </div>
                ${cmd.shortcut && cmd.shortcut.length > 0 ? `
                    <div class="command-shortcut">
                        ${cmd.shortcut.map(key => `<span class="key">${escapeHtml(key)}</span>`).join('')}
                    </div>
                ` : ''}
            `;

            item.addEventListener('click', () => executeCommand(globalIndex));
            item.addEventListener('mouseenter', () => {
                selectedIndex = globalIndex;
                renderCommands();
            });

            categoryDiv.appendChild(item);
        });

        commandList.appendChild(categoryDiv);
    });

    // Scroll selected item into view
    const selectedItem = commandList.querySelector('.command-item.selected');
    if (selectedItem) {
        selectedItem.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
    }
}

// Execute command
function executeCommand(index) {
    const cmd = filteredCommands[index];
    if (cmd && cmd.action) {
        cmd.action();
        closePalette();
    }
}

// Open palette
function openPalette() {
    overlay.classList.add('active');
    searchInput.value = '';
    searchInput.focus();
    selectedIndex = 0;
    filterCommands('');
}

// Close palette
function closePalette() {
    overlay.classList.remove('active');
}

// Keyboard event handling
document.addEventListener('keydown', (e) => {
    // Open palette with Ctrl+K or Cmd+K
    if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
        e.preventDefault();
        if (overlay.classList.contains('active')) {
            closePalette();
        } else {
            openPalette();
        }
        return;
    }

    // Handle keys when palette is open
    if (overlay.classList.contains('active')) {
        switch (e.key) {
            case 'Escape':
                e.preventDefault();
                closePalette();
                break;

            case 'ArrowDown':
                e.preventDefault();
                selectedIndex = Math.min(selectedIndex + 1, filteredCommands.length - 1);
                renderCommands();
                break;

            case 'ArrowUp':
                e.preventDefault();
                selectedIndex = Math.max(selectedIndex - 1, 0);
                renderCommands();
                break;

            case 'Enter':
                e.preventDefault();
                executeCommand(selectedIndex);
                break;
        }
    }
});

// Search input handler
searchInput.addEventListener('input', (e) => {
    selectedIndex = 0;
    filterCommands(e.target.value);
});

// Close on overlay click
overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
        closePalette();
    }
});

// Initial render
filterCommands('');
