// Encapsulation complète du composant Seed I.T
(() => {
  'use strict';

  // ============================================
  // Données de produits fictifs
  // ============================================
  const products = [
    { id: 1, name: 'Laptop Pro X1', category: 'electronique', price: 1299, emoji: '💻' },
    { id: 2, name: 'Smartphone Galaxy', category: 'electronique', price: 899, emoji: '📱' },
    { id: 3, name: 'T-shirt Premium', category: 'vetements', price: 45, emoji: '👕' },
    { id: 4, name: 'Jean Classique', category: 'vetements', price: 79, emoji: '👖' },
    { id: 5, name: 'Robe Élégante', category: 'vetements', price: 120, emoji: '👗' },
    { id: 6, name: 'Roman Fantastique', category: 'livres', price: 24, emoji: '📖' },
    { id: 7, name: 'Guide Pratique', category: 'livres', price: 18, emoji: '📚' },
    { id: 8, name: 'Ballon de Football', category: 'sport', price: 35, emoji: '⚽' },
    { id: 9, name: 'Raquette Tennis', category: 'sport', price: 150, emoji: '🎾' },
    { id: 10, name: 'Écouteurs Bluetooth', category: 'electronique', price: 129, emoji: '🎧' },
    { id: 11, name: 'Montre Connectée', category: 'electronique', price: 349, emoji: '⌚' },
    { id: 12, name: 'Baskets Running', category: 'sport', price: 95, emoji: '👟' }
  ];

  // ============================================
  // État du filtre
  // ============================================
  let currentCategory = 'all';
  let currentSearchText = '';
  let currentSort = 'name-asc';

  // ============================================
  // Éléments DOM
  // ============================================
  const searchInput = document.getElementById('searchInput');
  const sortSelect = document.getElementById('sortSelect');
  const categoryButtons = document.querySelectorAll('.seedit-filter-category');
  const productGrid = document.getElementById('productGrid');
  const itemCount = document.getElementById('itemCount');
  const emptyMessage = document.getElementById('emptyMessage');

  // ============================================
  // Initialisation
  // ============================================
  function init() {
    renderProducts();
    attachEventListeners();
  }

  // ============================================
  // Attacher les événements
  // ============================================
  function attachEventListeners() {
    // Recherche en temps réel
    searchInput.addEventListener('input', handleSearch);

    // Filtrage par catégorie
    categoryButtons.forEach(button => {
      button.addEventListener('click', handleCategoryClick);
      // Support clavier
      button.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          handleCategoryClick.call(button, e);
        }
      });
    });

    // Tri
    sortSelect.addEventListener('change', handleSortChange);
  }

  // ============================================
  // Gestion de la recherche textuelle
  // ============================================
  function handleSearch(e) {
    currentSearchText = e.target.value.toLowerCase().trim();
    renderProducts();
  }

  // ============================================
  // Gestion du clic sur une catégorie
  // ============================================
  function handleCategoryClick(e) {
    const button = e.currentTarget;
    currentCategory = button.dataset.category;

    // Mise à jour visuelle des boutons
    categoryButtons.forEach(btn => {
      btn.classList.remove('seedit-filter-category--active');
      btn.setAttribute('aria-pressed', 'false');
    });
    button.classList.add('seedit-filter-category--active');
    button.setAttribute('aria-pressed', 'true');

    renderProducts();
  }

  // ============================================
  // Gestion du changement de tri
  // ============================================
  function handleSortChange(e) {
    currentSort = e.target.value;
    renderProducts();
  }

  // ============================================
  // Filtrer les produits selon les critères
  // ============================================
  function filterProducts() {
    return products.filter(product => {
      // Filtre par catégorie
      const categoryMatch = currentCategory === 'all' || product.category === currentCategory;

      // Filtre par recherche textuelle
      const searchMatch = currentSearchText === '' ||
                         product.name.toLowerCase().includes(currentSearchText);

      return categoryMatch && searchMatch;
    });
  }

  // ============================================
  // Trier les produits
  // ============================================
  function sortProducts(productsToSort) {
    const sorted = [...productsToSort];

    switch (currentSort) {
      case 'name-asc':
        sorted.sort((a, b) => a.name.localeCompare(b.name));
        break;
      case 'name-desc':
        sorted.sort((a, b) => b.name.localeCompare(a.name));
        break;
      case 'price-asc':
        sorted.sort((a, b) => a.price - b.price);
        break;
      case 'price-desc':
        sorted.sort((a, b) => b.price - a.price);
        break;
    }

    return sorted;
  }

  // ============================================
  // Obtenir le nom français de la catégorie
  // ============================================
  function getCategoryLabel(category) {
    const labels = {
      'electronique': 'Électronique',
      'vetements': 'Vêtements',
      'livres': 'Livres',
      'sport': 'Sport'
    };
    return labels[category] || category;
  }

  // ============================================
  // Générer le HTML d'une carte produit
  // ============================================
  function createProductCard(product) {
    const card = document.createElement('article');
    card.className = 'seedit-filter-item';
    card.setAttribute('tabindex', '0');
    card.setAttribute('role', 'button');
    card.setAttribute('aria-label', `${product.name}, ${product.price}€`);

    card.innerHTML = `
      <div class="seedit-filter-item-image">${product.emoji}</div>
      <div class="seedit-filter-item-category">${getCategoryLabel(product.category)}</div>
      <h3 class="seedit-filter-item-name">${product.name}</h3>
      <div class="seedit-filter-item-price">${product.price}€</div>
    `;

    // Support du clavier pour l'interaction
    card.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        // Simulation d'action (exemple : ouvrir détail produit)
        card.style.transform = 'scale(0.95)';
        setTimeout(() => {
          card.style.transform = '';
        }, 150);
      }
    });

    return card;
  }

  // ============================================
  // Rendre les produits dans la grille
  // ============================================
  function renderProducts() {
    // Filtrer et trier
    const filteredProducts = filterProducts();
    const sortedProducts = sortProducts(filteredProducts);

    // Vider la grille
    productGrid.innerHTML = '';

    // Afficher les produits ou le message vide
    if (sortedProducts.length === 0) {
      productGrid.style.display = 'none';
      emptyMessage.removeAttribute('hidden');
    } else {
      productGrid.style.display = 'grid';
      emptyMessage.setAttribute('hidden', '');

      // Ajouter chaque carte avec un léger délai pour l'animation
      sortedProducts.forEach((product, index) => {
        const card = createProductCard(product);
        // Décalage de l'animation pour un effet cascade
        card.style.animationDelay = `${index * 0.05}s`;
        productGrid.appendChild(card);
      });
    }

    // Mettre à jour le compteur
    updateItemCount(sortedProducts.length);
  }

  // ============================================
  // Mettre à jour le compteur de résultats
  // ============================================
  function updateItemCount(count) {
    itemCount.textContent = count;
  }

  // ============================================
  // Lancement au chargement de la page
  // ============================================
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
