// ===================================
// Seed I.T - Shadow Box Animée
// Script pour animation dynamique des ombres
// ===================================

(() => {
  'use strict';

  // Initialisation au chargement du DOM
  document.addEventListener('DOMContentLoaded', initShadowBoxes);

  /**
   * Initialise toutes les shadow boxes de la page
   */
  function initShadowBoxes() {
    const shadowBoxes = document.querySelectorAll('.seedit-shadowbox');

    shadowBoxes.forEach(box => {
      // Événement mousemove : calcul de la position et mise à jour des ombres
      box.addEventListener('mousemove', handleMouseMove);

      // Événement mouseleave : réinitialisation des ombres
      box.addEventListener('mouseleave', handleMouseLeave);

      // Support tactile : simulation d'effet au touch
      box.addEventListener('touchstart', handleTouchStart, { passive: true });
      box.addEventListener('touchend', handleTouchEnd);
    });
  }

  /**
   * Gère le mouvement de la souris sur une shadow box
   * Calcule les coordonnées relatives et met à jour les CSS custom properties
   * @param {MouseEvent} event - Événement mousemove
   */
  function handleMouseMove(event) {
    const box = event.currentTarget;
    const rect = box.getBoundingClientRect();

    // Calcul de la position relative du curseur (centrée sur 0,0)
    const x = event.clientX - rect.left - rect.width / 2;
    const y = event.clientY - rect.top - rect.height / 2;

    // Normalisation de la position (entre -1 et 1)
    const normalizedX = x / (rect.width / 2);
    const normalizedY = y / (rect.height / 2);

    // Calcul de l'offset des ombres (maximum ±20px)
    const shadowX = normalizedX * 20;
    const shadowY = normalizedY * 20;

    // Mise à jour des CSS custom properties
    box.style.setProperty('--seedit-shadowbox-shadow-x', shadowX);
    box.style.setProperty('--seedit-shadowbox-shadow-y', shadowY);
  }

  /**
   * Réinitialise les ombres quand la souris quitte la shadow box
   * @param {MouseEvent} event - Événement mouseleave
   */
  function handleMouseLeave(event) {
    const box = event.currentTarget;

    // Retour progressif à la position neutre
    box.style.setProperty('--seedit-shadowbox-shadow-x', '0');
    box.style.setProperty('--seedit-shadowbox-shadow-y', '0');
  }

  /**
   * Gère le début d'un toucher tactile
   * Active l'effet d'ombre au centre
   * @param {TouchEvent} event - Événement touchstart
   */
  function handleTouchStart(event) {
    const box = event.currentTarget;

    // Simulation d'un léger effet d'ombre au centre
    box.style.setProperty('--seedit-shadowbox-shadow-x', '5');
    box.style.setProperty('--seedit-shadowbox-shadow-y', '5');
  }

  /**
   * Gère la fin d'un toucher tactile
   * Réinitialise les ombres
   * @param {TouchEvent} event - Événement touchend
   */
  function handleTouchEnd(event) {
    const box = event.currentTarget;

    // Retour à la position neutre
    box.style.setProperty('--seedit-shadowbox-shadow-x', '0');
    box.style.setProperty('--seedit-shadowbox-shadow-y', '0');
  }

})();
