// ===================================
// Seed I.T - Composant de Notation Étoiles
// Script pour gestion interactive des notes
// ===================================

(() => {
  'use strict';

  // Initialisation au chargement du DOM
  document.addEventListener('DOMContentLoaded', initRatings);

  /**
   * Initialise tous les composants de notation de la page
   */
  function initRatings() {
    const ratingContainers = document.querySelectorAll('.seedit-rating');

    ratingContainers.forEach(container => {
      const mode = container.getAttribute('data-mode');
      const initialValue = parseFloat(container.getAttribute('data-initial-value')) || 0;

      if (mode === 'interactive') {
        initInteractiveRating(container, initialValue);
      } else if (mode === 'readonly') {
        initReadonlyRating(container, initialValue);
      }
    });
  }

  /**
   * Initialise un composant de notation interactif
   * @param {HTMLElement} container - Conteneur de notation
   * @param {number} initialValue - Note initiale (0-5)
   */
  function initInteractiveRating(container, initialValue) {
    const stars = container.querySelectorAll('.seedit-rating__star');
    let currentRating = initialValue;

    // Appliquer la note initiale
    updateStarsDisplay(stars, currentRating);

    // Événement : clic sur une étoile
    stars.forEach(star => {
      star.addEventListener('click', () => {
        const value = parseInt(star.getAttribute('data-value'), 10);
        currentRating = value;
        updateStarsDisplay(stars, currentRating);
        emitRatingChange(container, currentRating);
        updateResultDisplay(container, currentRating);
      });

      // Événement : survol d'une étoile
      star.addEventListener('mouseenter', () => {
        const value = parseInt(star.getAttribute('data-value'), 10);
        highlightStars(stars, value);
      });
    });

    // Événement : sortie du survol du conteneur
    container.addEventListener('mouseleave', () => {
      updateStarsDisplay(stars, currentRating);
    });

    // Gestion du clavier (accessibilité)
    container.addEventListener('keydown', (event) => {
      handleKeyboardNavigation(event, stars, currentRating, (newRating) => {
        currentRating = newRating;
        updateStarsDisplay(stars, currentRating);
        emitRatingChange(container, currentRating);
        updateResultDisplay(container, currentRating);
      });
    });

    // Mettre à jour l'affichage initial
    updateResultDisplay(container, currentRating);
  }

  /**
   * Initialise un composant de notation en lecture seule
   * @param {HTMLElement} container - Conteneur de notation
   * @param {number} value - Note à afficher (0-5, supporte les décimales)
   */
  function initReadonlyRating(container, value) {
    const stars = container.querySelectorAll('.seedit-rating__star');

    stars.forEach((star, index) => {
      const starValue = index + 1;

      if (starValue <= Math.floor(value)) {
        // Étoile complète
        star.classList.add('seedit-rating__star--filled');
      } else if (starValue === Math.ceil(value) && value % 1 !== 0) {
        // Demi-étoile
        star.classList.add('seedit-rating__star--half');
      }
    });
  }

  /**
   * Met à jour l'affichage visuel des étoiles
   * @param {NodeList} stars - Liste des éléments étoile
   * @param {number} rating - Note actuelle
   */
  function updateStarsDisplay(stars, rating) {
    stars.forEach((star, index) => {
      const starValue = index + 1;

      if (starValue <= rating) {
        star.classList.add('seedit-rating__star--filled');
        star.setAttribute('aria-checked', 'true');
      } else {
        star.classList.remove('seedit-rating__star--filled');
        star.setAttribute('aria-checked', 'false');
      }

      // Retirer les classes de survol
      star.classList.remove('seedit-rating__star--hover');
    });
  }

  /**
   * Surligne les étoiles au survol
   * @param {NodeList} stars - Liste des éléments étoile
   * @param {number} hoverValue - Valeur survolée
   */
  function highlightStars(stars, hoverValue) {
    stars.forEach((star, index) => {
      const starValue = index + 1;

      if (starValue <= hoverValue) {
        star.classList.add('seedit-rating__star--hover');
      } else {
        star.classList.remove('seedit-rating__star--hover');
      }
    });
  }

  /**
   * Gère la navigation au clavier
   * @param {KeyboardEvent} event - Événement clavier
   * @param {NodeList} stars - Liste des éléments étoile
   * @param {number} currentRating - Note actuelle
   * @param {Function} callback - Fonction de callback pour mettre à jour la note
   */
  function handleKeyboardNavigation(event, stars, currentRating, callback) {
    let newRating = currentRating;

    switch (event.key) {
      case 'ArrowRight':
      case 'ArrowUp':
        event.preventDefault();
        newRating = Math.min(currentRating + 1, stars.length);
        break;

      case 'ArrowLeft':
      case 'ArrowDown':
        event.preventDefault();
        newRating = Math.max(currentRating - 1, 0);
        break;

      case 'Home':
        event.preventDefault();
        newRating = 1;
        break;

      case 'End':
        event.preventDefault();
        newRating = stars.length;
        break;

      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
        event.preventDefault();
        newRating = parseInt(event.key, 10);
        break;

      default:
        return;
    }

    if (newRating !== currentRating) {
      callback(newRating);
    }
  }

  /**
   * Émet un événement personnalisé lors du changement de note
   * @param {HTMLElement} container - Conteneur de notation
   * @param {number} rating - Nouvelle note
   */
  function emitRatingChange(container, rating) {
    const event = new CustomEvent('seedit-rating-change', {
      detail: { rating },
      bubbles: true
    });
    container.dispatchEvent(event);
  }

  /**
   * Met à jour l'affichage textuel du résultat
   * @param {HTMLElement} container - Conteneur de notation
   * @param {number} rating - Note actuelle
   */
  function updateResultDisplay(container, rating) {
    const section = container.closest('.seedit-rating-demo__section');
    if (!section) return;

    const resultElement = section.querySelector('.seedit-rating-demo__result');
    if (resultElement && resultElement.id === 'result-interactive') {
      resultElement.textContent = `Note actuelle : ${rating}/5`;
    }
  }

  // Événement global pour écouter les changements de notation
  document.addEventListener('seedit-rating-change', (event) => {
    // Exemple d'utilisation : log dans la console (à retirer en production)
    // console.log('Note changée :', event.detail.rating);
  });

})();
