// ===================================
// Seed I.T - Effet Machine à Écrire
// Script pour animation de texte caractère par caractère
// ===================================

(() => {
  'use strict';

  // Initialisation au chargement du DOM
  document.addEventListener('DOMContentLoaded', initTypewriters);

  /**
   * Initialise tous les composants machine à écrire de la page
   */
  function initTypewriters() {
    const typewriters = document.querySelectorAll('.seedit-typewriter');

    typewriters.forEach(typewriter => {
      const textElement = typewriter.querySelector('.seedit-typewriter__text');
      if (!textElement) return;

      // Récupération de la configuration
      const config = getTypewriterConfig(typewriter);

      // Démarrer l'animation
      startTypewriter(textElement, config);
    });
  }

  /**
   * Récupère la configuration depuis les data-attributes
   * @param {HTMLElement} element - Élément typewriter
   * @returns {Object} Configuration
   */
  function getTypewriterConfig(element) {
    // Texte unique ou phrases multiples
    const singleText = element.getAttribute('data-text');
    const multiTexts = element.getAttribute('data-texts');

    let texts = [];
    if (multiTexts) {
      try {
        texts = JSON.parse(multiTexts);
      } catch (e) {
        console.error('Erreur de parsing data-texts:', e);
        texts = [singleText || 'Texte par défaut'];
      }
    } else {
      texts = [singleText || 'Texte par défaut'];
    }

    return {
      texts: texts,
      speed: parseInt(element.getAttribute('data-speed'), 10) || 80,
      cursor: element.getAttribute('data-cursor') !== 'false',
      loop: element.getAttribute('data-loop') === 'true',
      loopDelay: parseInt(element.getAttribute('data-loop-delay'), 10) || 2000,
      deleteSpeed: parseInt(element.getAttribute('data-delete-speed'), 10) || 40
    };
  }

  /**
   * Démarre l'animation de machine à écrire
   * @param {HTMLElement} textElement - Élément texte à animer
   * @param {Object} config - Configuration
   */
  function startTypewriter(textElement, config) {
    let currentTextIndex = 0;
    let currentCharIndex = 0;
    let isDeleting = false;
    let timeoutId = null;

    /**
     * Fonction principale d'animation (récursive)
     */
    function type() {
      const currentText = config.texts[currentTextIndex];

      if (isDeleting) {
        // Mode suppression : retirer un caractère
        currentCharIndex--;
        textElement.textContent = currentText.substring(0, currentCharIndex);

        if (currentCharIndex === 0) {
          // Suppression terminée, passer au texte suivant
          isDeleting = false;
          currentTextIndex = (currentTextIndex + 1) % config.texts.length;

          // Si pas de boucle et on a fini tous les textes, arrêter
          if (!config.loop && currentTextIndex === 0 && config.texts.length > 1) {
            return;
          }

          // Petite pause avant de commencer le nouveau texte
          timeoutId = setTimeout(type, 500);
          return;
        }
      } else {
        // Mode écriture : ajouter un caractère
        currentCharIndex++;
        textElement.textContent = currentText.substring(0, currentCharIndex);

        if (currentCharIndex === currentText.length) {
          // Écriture terminée

          // Si c'est le dernier texte et pas de boucle, arrêter
          if (!config.loop && currentTextIndex === config.texts.length - 1) {
            return;
          }

          // Si plusieurs textes ou mode boucle, préparer la suppression
          if (config.texts.length > 1 || config.loop) {
            isDeleting = true;
            // Pause avant de commencer à supprimer
            timeoutId = setTimeout(type, config.loopDelay);
            return;
          }

          // Sinon, on s'arrête
          return;
        }
      }

      // Continuer l'animation
      const speed = isDeleting ? config.deleteSpeed : config.speed;
      timeoutId = setTimeout(type, speed);
    }

    // Démarrer l'animation
    type();

    // Nettoyer le timeout quand l'élément est retiré du DOM
    const observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        mutation.removedNodes.forEach((node) => {
          if (node === textElement.parentElement) {
            if (timeoutId) {
              clearTimeout(timeoutId);
            }
            observer.disconnect();
          }
        });
      });
    });

    if (textElement.parentElement) {
      observer.observe(textElement.parentElement.parentElement, {
        childList: true,
        subtree: true
      });
    }
  }

})();
