// ===================================
// Seed I.T - Bouton Magnétique
// Script pour effet d'attraction magnétique au survol
// ===================================

(() => {
  'use strict';

  // Initialisation au chargement du DOM
  document.addEventListener('DOMContentLoaded', initMagneticButtons);

  /**
   * Initialise tous les boutons magnétiques de la page
   */
  function initMagneticButtons() {
    const magneticButtons = document.querySelectorAll('[data-magnetic="true"]');

    magneticButtons.forEach(button => {
      // Récupération de la configuration
      const config = {
        strength: parseFloat(button.getAttribute('data-magnetic-strength')) || 0.3,
        distance: parseInt(button.getAttribute('data-magnetic-distance'), 10) || 150,
        rotation: button.getAttribute('data-magnetic-rotation') === 'true'
      };

      // Variables d'état pour l'animation
      let isHovering = false;
      let currentX = 0;
      let currentY = 0;
      let targetX = 0;
      let targetY = 0;
      let animationFrame = null;

      /**
       * Calcule la position magnétique en fonction de la souris
       */
      function calculateMagneticEffect(event) {
        const rect = button.getBoundingClientRect();
        const centerX = rect.left + rect.width / 2;
        const centerY = rect.top + rect.height / 2;

        // Position de la souris
        const mouseX = event.clientX;
        const mouseY = event.clientY;

        // Distance entre la souris et le centre du bouton
        const deltaX = mouseX - centerX;
        const deltaY = mouseY - centerY;
        const distance = Math.sqrt(deltaX * deltaX + deltaY * deltaY);

        // Si la souris est dans le rayon d'action
        if (distance < config.distance) {
          // Facteur d'attraction (1 = très près, 0 = loin)
          const attractionFactor = 1 - (distance / config.distance);

          // Calcul du déplacement magnétique
          targetX = deltaX * config.strength * attractionFactor;
          targetY = deltaY * config.strength * attractionFactor;
        } else {
          // Retour à la position neutre
          targetX = 0;
          targetY = 0;
        }
      }

      /**
       * Animation fluide avec interpolation (lerp)
       */
      function animate() {
        // Interpolation linéaire pour un mouvement fluide
        const lerpFactor = 0.15;
        currentX += (targetX - currentX) * lerpFactor;
        currentY += (targetY - currentY) * lerpFactor;

        // Appliquer le transform
        let transform = `translate(${currentX}px, ${currentY}px)`;

        // Ajouter une rotation si activée
        if (config.rotation) {
          const rotationX = currentY * 0.05;
          const rotationY = -currentX * 0.05;
          transform += ` perspective(1000px) rotateX(${rotationX}deg) rotateY(${rotationY}deg)`;
        }

        button.style.transform = transform;

        // Continuer l'animation si nécessaire
        if (isHovering || Math.abs(targetX - currentX) > 0.1 || Math.abs(targetY - currentY) > 0.1) {
          animationFrame = requestAnimationFrame(animate);
        }
      }

      /**
       * Gestion de l'événement mousemove
       */
      function handleMouseMove(event) {
        if (!isHovering) return;
        calculateMagneticEffect(event);
      }

      /**
       * Gestion de l'événement mouseenter
       */
      function handleMouseEnter(event) {
        isHovering = true;
        calculateMagneticEffect(event);

        // Démarrer l'animation si elle n'est pas déjà en cours
        if (!animationFrame) {
          animate();
        }
      }

      /**
       * Gestion de l'événement mouseleave
       */
      function handleMouseLeave() {
        isHovering = false;
        targetX = 0;
        targetY = 0;

        // L'animation continuera jusqu'à ce que le bouton revienne à la position neutre
      }

      // Attacher les événements
      button.addEventListener('mouseenter', handleMouseEnter);
      button.addEventListener('mousemove', handleMouseMove);
      button.addEventListener('mouseleave', handleMouseLeave);

      // Nettoyage quand l'élément est retiré du DOM
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          mutation.removedNodes.forEach((node) => {
            if (node === button) {
              button.removeEventListener('mouseenter', handleMouseEnter);
              button.removeEventListener('mousemove', handleMouseMove);
              button.removeEventListener('mouseleave', handleMouseLeave);
              if (animationFrame) {
                cancelAnimationFrame(animationFrame);
              }
              observer.disconnect();
            }
          });
        });
      });

      if (button.parentElement) {
        observer.observe(button.parentElement, {
          childList: true,
          subtree: true
        });
      }
    });
  }

})();
