/* ===================================
   Seed I.T - Input Numérique avec +/- Buttons
   Gestion des interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise un input numérique avec ses boutons +/-
   * @param {HTMLElement} container - Conteneur .seedit-number-input
   */
  function initNumberInput(container) {
    const input = container.querySelector('.seedit-number-input__field');
    const btnMinus = container.querySelector('.seedit-number-input__btn--minus');
    const btnPlus = container.querySelector('.seedit-number-input__btn--plus');

    if (!input || !btnMinus || !btnPlus) return;

    // Récupérer les limites depuis les data attributes
    const min = parseFloat(container.dataset.min) || 0;
    const max = parseFloat(container.dataset.max) || Infinity;
    const step = parseFloat(container.dataset.step) || 1;

    /**
     * Met à jour l'état des boutons selon la valeur actuelle
     */
    function updateButtonStates() {
      const value = parseFloat(input.value) || min;

      // Désactiver le bouton - si on est à la valeur minimale
      btnMinus.disabled = value <= min;

      // Désactiver le bouton + si on est à la valeur maximale
      btnPlus.disabled = value >= max;
    }

    /**
     * Décrémente la valeur
     */
    function decrement() {
      // Vérifier si le composant est désactivé
      if (input.disabled || container.classList.contains('seedit-number-input--disabled')) {
        return;
      }

      const currentValue = parseFloat(input.value) || min;
      const newValue = Math.max(min, currentValue - step);

      input.value = newValue;
      updateButtonStates();

      // Déclencher un événement change pour permettre l'écoute externe
      input.dispatchEvent(new Event('change', { bubbles: true }));
    }

    /**
     * Incrémente la valeur
     */
    function increment() {
      // Vérifier si le composant est désactivé
      if (input.disabled || container.classList.contains('seedit-number-input--disabled')) {
        return;
      }

      const currentValue = parseFloat(input.value) || min;
      const newValue = Math.min(max, currentValue + step);

      input.value = newValue;
      updateButtonStates();

      // Déclencher un événement change pour permettre l'écoute externe
      input.dispatchEvent(new Event('change', { bubbles: true }));
    }

    /**
     * Valide la saisie manuelle dans l'input
     */
    function validateInput() {
      let value = parseFloat(input.value);

      // Si la valeur n'est pas un nombre, remettre le min
      if (isNaN(value)) {
        input.value = min;
        updateButtonStates();
        return;
      }

      // Contraindre la valeur dans les limites min/max
      if (value < min) {
        input.value = min;
      } else if (value > max) {
        input.value = max;
      }

      updateButtonStates();
    }

    // Écouteurs d'événements pour les boutons
    btnMinus.addEventListener('click', decrement);
    btnPlus.addEventListener('click', increment);

    // Validation lors de la modification manuelle de l'input
    input.addEventListener('input', updateButtonStates);
    input.addEventListener('blur', validateInput);

    // Support clavier : flèches haut/bas
    input.addEventListener('keydown', (e) => {
      // Ne pas traiter les touches si le composant est désactivé
      if (input.disabled || container.classList.contains('seedit-number-input--disabled')) {
        return;
      }

      if (e.key === 'ArrowUp') {
        e.preventDefault();
        increment();
      } else if (e.key === 'ArrowDown') {
        e.preventDefault();
        decrement();
      }
    });

    // Initialiser l'état des boutons au chargement
    updateButtonStates();
  }

  /**
   * Initialise le calcul du total pour l'exemple panier e-commerce
   */
  function initCartTotal() {
    const cartItem = document.querySelector('.seedit-cart-item');
    if (!cartItem) return;

    const quantityInput = cartItem.querySelector('.seedit-number-input__field');
    const totalAmountElement = cartItem.querySelector('.seedit-cart-item__total-amount');
    const priceText = cartItem.querySelector('.seedit-cart-item__price');

    if (!quantityInput || !totalAmountElement || !priceText) return;

    // Extraire le prix unitaire du texte (ex: "149€" -> 149)
    const unitPrice = parseFloat(priceText.textContent.replace(/[^0-9.]/g, ''));

    if (isNaN(unitPrice)) return;

    /**
     * Met à jour le total affiché
     */
    function updateTotal() {
      const quantity = parseFloat(quantityInput.value) || 1;
      const total = unitPrice * quantity;

      // Mettre à jour l'affichage avec formatage
      totalAmountElement.textContent = `${total}€`;
    }

    // Écouter les changements de quantité
    quantityInput.addEventListener('change', updateTotal);
    quantityInput.addEventListener('input', updateTotal);

    // Initialiser le total au chargement
    updateTotal();
  }

  /**
   * Initialisation de tous les inputs numériques au chargement du DOM
   */
  function init() {
    const numberInputs = document.querySelectorAll('.seedit-number-input');

    numberInputs.forEach(input => {
      initNumberInput(input);
    });

    // Initialiser le calcul du panier si présent
    initCartTotal();
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer la fonction d'initialisation pour usage externe si nécessaire
  window.SeeditNumberInput = {
    init: initNumberInput
  };

})();
