/* ===================================
   Seed I.T - Système de Votes Up/Down
   Gestion des interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise un système de vote
   * @param {HTMLElement} container - Conteneur .seedit-vote
   */
  function initVote(container) {
    const btnUp = container.querySelector('.seedit-vote__btn--up');
    const btnDown = container.querySelector('.seedit-vote__btn--down');
    const countElement = container.querySelector('.seedit-vote__count');

    if (!btnUp || !btnDown || !countElement) return;

    // Récupérer le score initial et l'état du vote depuis les data attributes
    const initialScore = parseInt(container.dataset.initialScore) || 0;
    let currentScore = initialScore;
    let userVote = container.dataset.userVote || null; // 'up', 'down', ou null

    // Initialiser l'affichage et les classes actives
    updateDisplay();

    /**
     * Met à jour l'affichage du compteur et des boutons
     */
    function updateDisplay() {
      // Mettre à jour le texte du compteur
      countElement.textContent = currentScore;

      // Mettre à jour les classes actives des boutons
      if (userVote === 'up') {
        btnUp.classList.add('seedit-vote__btn--active');
        btnDown.classList.remove('seedit-vote__btn--active');
        container.classList.add('seedit-vote--upvoted');
        container.classList.remove('seedit-vote--downvoted');
        btnUp.setAttribute('aria-label', 'Retirer le vote pour');
      } else if (userVote === 'down') {
        btnDown.classList.add('seedit-vote__btn--active');
        btnUp.classList.remove('seedit-vote__btn--active');
        container.classList.add('seedit-vote--downvoted');
        container.classList.remove('seedit-vote--upvoted');
        btnDown.setAttribute('aria-label', 'Retirer le vote contre');
      } else {
        btnUp.classList.remove('seedit-vote__btn--active');
        btnDown.classList.remove('seedit-vote__btn--active');
        container.classList.remove('seedit-vote--upvoted', 'seedit-vote--downvoted');
        btnUp.setAttribute('aria-label', 'Voter pour');
        btnDown.setAttribute('aria-label', 'Voter contre');
      }
    }

    /**
     * Anime le compteur lors d'un changement
     */
    function animateCount() {
      countElement.classList.add('seedit-vote__count--updating');
      setTimeout(() => {
        countElement.classList.remove('seedit-vote__count--updating');
      }, 400);
    }

    /**
     * Gère le clic sur le bouton upvote
     */
    function handleUpvote() {
      // Vérifier si le composant est désactivé
      if (btnUp.disabled || container.classList.contains('seedit-vote--disabled')) {
        return;
      }

      if (userVote === 'up') {
        // Annuler l'upvote
        currentScore--;
        userVote = null;
      } else if (userVote === 'down') {
        // Changer de downvote à upvote
        currentScore += 2;
        userVote = 'up';
      } else {
        // Nouveau upvote
        currentScore++;
        userVote = 'up';
      }

      animateCount();
      updateDisplay();

      // Déclencher un événement personnalisé pour permettre l'écoute externe
      container.dispatchEvent(new CustomEvent('vote-change', {
        bubbles: true,
        detail: {
          vote: userVote,
          score: currentScore,
          initialScore: initialScore
        }
      }));
    }

    /**
     * Gère le clic sur le bouton downvote
     */
    function handleDownvote() {
      // Vérifier si le composant est désactivé
      if (btnDown.disabled || container.classList.contains('seedit-vote--disabled')) {
        return;
      }

      if (userVote === 'down') {
        // Annuler le downvote
        currentScore++;
        userVote = null;
      } else if (userVote === 'up') {
        // Changer d'upvote à downvote
        currentScore -= 2;
        userVote = 'down';
      } else {
        // Nouveau downvote
        currentScore--;
        userVote = 'down';
      }

      animateCount();
      updateDisplay();

      // Déclencher un événement personnalisé pour permettre l'écoute externe
      container.dispatchEvent(new CustomEvent('vote-change', {
        bubbles: true,
        detail: {
          vote: userVote,
          score: currentScore,
          initialScore: initialScore
        }
      }));
    }

    // Écouter les clics sur les boutons
    btnUp.addEventListener('click', handleUpvote);
    btnDown.addEventListener('click', handleDownvote);

    // Support clavier : flèches haut/bas (quand le focus est sur un bouton)
    container.addEventListener('keydown', (e) => {
      // Ne pas traiter les touches si le composant est désactivé
      if (container.classList.contains('seedit-vote--disabled')) {
        return;
      }

      // Upvote avec flèche haut
      if (e.key === 'ArrowUp' && document.activeElement === btnUp) {
        e.preventDefault();
        handleUpvote();
      }
      // Downvote avec flèche bas
      else if (e.key === 'ArrowDown' && document.activeElement === btnDown) {
        e.preventDefault();
        handleDownvote();
      }
    });
  }

  /**
   * Initialisation de tous les systèmes de vote au chargement du DOM
   */
  function init() {
    const voteContainers = document.querySelectorAll('.seedit-vote');

    voteContainers.forEach(container => {
      initVote(container);
    });
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer la fonction d'initialisation pour usage externe si nécessaire
  window.SeeditVote = {
    init: initVote
  };

})();
