/* ===================================
   Seed I.T - Tags Cliquables
   Gestion des interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise un tag cliquable
   * @param {HTMLElement} tag - Élément tag (button ou div)
   */
  function initTag(tag) {
    // Si c'est un tag avec suppression (div), gérer le bouton de suppression
    if (tag.classList.contains('seedit-tag--removable')) {
      const removeBtn = tag.querySelector('.seedit-tag__remove');
      if (removeBtn) {
        removeBtn.addEventListener('click', (e) => {
          e.stopPropagation(); // Empêcher le toggle du tag
          handleTagRemove(tag);
        });
      }
      return; // Les tags removable ne sont pas cliquables pour le toggle
    }

    // Pour les tags button simples, gérer le toggle
    if (tag.tagName === 'BUTTON') {
      tag.addEventListener('click', () => {
        handleTagToggle(tag);
      });

      // Support clavier : Space et Enter pour toggle
      tag.addEventListener('keydown', (e) => {
        if (e.key === ' ' || e.key === 'Enter') {
          e.preventDefault();
          handleTagToggle(tag);
        }
      });
    }
  }

  /**
   * Gère le toggle d'un tag (actif/inactif)
   * @param {HTMLElement} tag - Élément tag
   */
  function handleTagToggle(tag) {
    // Vérifier si le tag est désactivé
    if (tag.disabled) return;

    // Toggle la classe active
    const wasActive = tag.classList.contains('seedit-tag--active');
    tag.classList.toggle('seedit-tag--active');
    const isActive = !wasActive;

    // Déclencher un événement personnalisé
    tag.dispatchEvent(new CustomEvent('tag-toggle', {
      bubbles: true,
      detail: {
        value: tag.dataset.value || tag.textContent.trim(),
        active: isActive,
        tag: tag
      }
    }));
  }

  /**
   * Gère la suppression d'un tag
   * @param {HTMLElement} tag - Élément tag à supprimer
   */
  function handleTagRemove(tag) {
    // Animation de sortie
    tag.style.transition = 'all 0.3s ease';
    tag.style.opacity = '0';
    tag.style.transform = 'scale(0.8)';

    // Déclencher un événement personnalisé avant la suppression
    tag.dispatchEvent(new CustomEvent('tag-remove', {
      bubbles: true,
      detail: {
        value: tag.dataset.value || tag.querySelector('.seedit-tag__label')?.textContent.trim(),
        filterGroup: tag.dataset.filterGroup,
        tag: tag
      }
    }));

    // Supprimer l'élément après l'animation
    setTimeout(() => {
      tag.remove();
    }, 300);
  }

  /**
   * Initialise le système de filtrage
   */
  function initFilterSystem() {
    const filterSystem = document.querySelector('.seedit-filter-system');
    if (!filterSystem) return;

    const clearAllBtn = filterSystem.querySelector('.seedit-filter-system__clear');
    const activeFiltersContainer = document.getElementById('active-filters');
    const resultsCount = filterSystem.querySelector('.seedit-filter-system__count strong');

    // Gérer le bouton "Tout effacer"
    if (clearAllBtn) {
      clearAllBtn.addEventListener('click', () => {
        // Désactiver tous les tags de filtres
        const allFilterTags = filterSystem.querySelectorAll('[data-filter-group] .seedit-tag--active');
        allFilterTags.forEach(tag => {
          if (tag.tagName === 'BUTTON') {
            tag.classList.remove('seedit-tag--active');
          }
        });

        // Supprimer tous les tags actifs
        if (activeFiltersContainer) {
          const activeTags = activeFiltersContainer.querySelectorAll('.seedit-tag--removable');
          activeTags.forEach(tag => {
            tag.style.transition = 'all 0.3s ease';
            tag.style.opacity = '0';
            tag.style.transform = 'scale(0.8)';
          });

          setTimeout(() => {
            activeTags.forEach(tag => tag.remove());
            updateResultsCount();
          }, 300);
        }
      });
    }

    // Gérer les toggles de tags dans les groupes de filtres
    const filterGroups = filterSystem.querySelectorAll('[data-filter-group]');
    filterGroups.forEach(group => {
      group.addEventListener('tag-toggle', (e) => {
        const detail = e.detail;
        const filterGroup = group.dataset.filterGroup;

        if (detail.active) {
          // Ajouter le tag aux filtres actifs
          addActiveFilterTag(filterGroup, detail.value, detail.tag);
        } else {
          // Retirer le tag des filtres actifs
          removeActiveFilterTag(filterGroup, detail.value);
        }

        updateResultsCount();
      });
    });

    // Gérer la suppression des tags actifs
    if (activeFiltersContainer) {
      activeFiltersContainer.addEventListener('tag-remove', (e) => {
        const detail = e.detail;

        // Désactiver le tag correspondant dans le groupe de filtres
        const filterGroup = filterSystem.querySelector(`[data-filter-group="${detail.filterGroup}"]`);
        if (filterGroup) {
          const filterTag = filterGroup.querySelector(`[data-value="${detail.value}"]`);
          if (filterTag && filterTag.tagName === 'BUTTON') {
            filterTag.classList.remove('seedit-tag--active');
          }
        }

        updateResultsCount();
      });
    }

    /**
     * Ajoute un tag aux filtres actifs
     */
    function addActiveFilterTag(filterGroup, value, originalTag) {
      if (!activeFiltersContainer) return;

      // Vérifier si le tag n'existe pas déjà
      const existing = activeFiltersContainer.querySelector(`[data-filter-group="${filterGroup}"][data-value="${value}"]`);
      if (existing) return;

      // Créer le tag actif
      const activeTag = document.createElement('div');
      activeTag.className = 'seedit-tag seedit-tag--removable seedit-tag--active';
      activeTag.dataset.filterGroup = filterGroup;
      activeTag.dataset.value = value;

      // Copier les classes de variante du tag original
      if (originalTag.classList.contains('seedit-tag--primary')) {
        activeTag.classList.add('seedit-tag--primary');
      } else if (originalTag.classList.contains('seedit-tag--success')) {
        activeTag.classList.add('seedit-tag--success');
      } else if (originalTag.classList.contains('seedit-tag--warning')) {
        activeTag.classList.add('seedit-tag--warning');
      } else if (originalTag.classList.contains('seedit-tag--danger')) {
        activeTag.classList.add('seedit-tag--danger');
      }

      const label = document.createElement('span');
      label.className = 'seedit-tag__label';
      label.textContent = originalTag.textContent.trim();

      const removeBtn = document.createElement('button');
      removeBtn.type = 'button';
      removeBtn.className = 'seedit-tag__remove';
      removeBtn.setAttribute('aria-label', `Supprimer le filtre ${label.textContent}`);

      const icon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
      icon.setAttribute('class', 'seedit-tag__remove-icon');
      icon.setAttribute('width', '12');
      icon.setAttribute('height', '12');
      icon.setAttribute('viewBox', '0 0 12 12');
      icon.setAttribute('fill', 'none');
      icon.setAttribute('aria-hidden', 'true');

      const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
      path.setAttribute('d', 'M9 3L3 9M3 3L9 9');
      path.setAttribute('stroke', 'currentColor');
      path.setAttribute('stroke-width', '2');
      path.setAttribute('stroke-linecap', 'round');
      path.setAttribute('stroke-linejoin', 'round');

      icon.appendChild(path);
      removeBtn.appendChild(icon);

      activeTag.appendChild(label);
      activeTag.appendChild(removeBtn);

      // Ajouter avec animation
      activeTag.style.opacity = '0';
      activeTag.style.transform = 'scale(0.8)';
      activeFiltersContainer.appendChild(activeTag);

      // Initialiser le tag (pour le bouton de suppression)
      initTag(activeTag);

      // Animer l'entrée
      setTimeout(() => {
        activeTag.style.transition = 'all 0.3s ease';
        activeTag.style.opacity = '1';
        activeTag.style.transform = 'scale(1)';
      }, 10);
    }

    /**
     * Retire un tag des filtres actifs
     */
    function removeActiveFilterTag(filterGroup, value) {
      if (!activeFiltersContainer) return;

      const activeTag = activeFiltersContainer.querySelector(`[data-filter-group="${filterGroup}"][data-value="${value}"]`);
      if (activeTag) {
        activeTag.style.transition = 'all 0.3s ease';
        activeTag.style.opacity = '0';
        activeTag.style.transform = 'scale(0.8)';

        setTimeout(() => {
          activeTag.remove();
        }, 300);
      }
    }

    /**
     * Met à jour le nombre de résultats (simulation)
     */
    function updateResultsCount() {
      if (!resultsCount || !activeFiltersContainer) return;

      const activeFiltersCount = activeFiltersContainer.querySelectorAll('.seedit-tag--removable').length;

      // Simulation : diminuer le nombre de résultats avec plus de filtres
      const baseCount = 156;
      const count = Math.max(1, Math.floor(baseCount / (activeFiltersCount + 1)));

      resultsCount.textContent = `${count} résultat${count > 1 ? 's' : ''}`;
    }

    // Initialiser le compte au chargement
    updateResultsCount();
  }

  /**
   * Initialisation de tous les tags au chargement du DOM
   */
  function init() {
    const tags = document.querySelectorAll('.seedit-tag');

    tags.forEach(tag => {
      initTag(tag);
    });

    // Initialiser le système de filtrage si présent
    initFilterSystem();
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer la fonction d'initialisation pour usage externe si nécessaire
  window.SeeditTag = {
    init: initTag
  };

})();
