/* ===================================
   Seed I.T - Formulaire de Contact Minimal
   Gestion des interactions et validation
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise un formulaire de contact
   * @param {HTMLFormElement} form - Élément form
   */
  function initContactForm(form) {
    if (!form) return;

    const container = form.closest('.seedit-contact-form');
    if (!container) return;

    // Gérer la soumission du formulaire
    form.addEventListener('submit', (e) => {
      e.preventDefault();
      handleSubmit(form, container);
    });

    // Gérer les champs pour retirer les erreurs lors de la saisie
    const fields = form.querySelectorAll('.seedit-contact-form__input, .seedit-contact-form__textarea');
    fields.forEach(field => {
      field.addEventListener('input', () => {
        const fieldContainer = field.closest('.seedit-contact-form__field');
        if (fieldContainer) {
          clearFieldError(fieldContainer);
        }
      });
    });

    // Gérer le bouton de retour depuis l'écran de succès
    const successBtn = container.querySelector('.seedit-contact-form__success-btn');
    if (successBtn) {
      successBtn.addEventListener('click', () => {
        resetForm(form, container);
      });
    }
  }

  /**
   * Gère la soumission du formulaire
   * @param {HTMLFormElement} form - Élément form
   * @param {HTMLElement} container - Conteneur du formulaire
   */
  function handleSubmit(form, container) {
    // Nettoyer les erreurs existantes
    clearAllErrors(form);

    // Valider le formulaire
    const validation = validateForm(form);

    if (!validation.isValid) {
      // Afficher les erreurs
      displayErrors(form, validation.errors);

      // Déclencher un événement d'erreur de validation
      container.dispatchEvent(new CustomEvent('contact-validation-error', {
        bubbles: true,
        detail: {
          errors: validation.errors
        }
      }));

      return;
    }

    // Récupérer les données du formulaire
    const formData = getFormData(form);

    // Déclencher un événement avant la soumission
    const beforeSubmitEvent = new CustomEvent('contact-before-submit', {
      bubbles: true,
      cancelable: true,
      detail: {
        formData: formData,
        form: form
      }
    });

    if (!container.dispatchEvent(beforeSubmitEvent)) {
      // Si l'événement est annulé, arrêter la soumission
      return;
    }

    // Afficher l'état de chargement
    showLoadingState(form, container);

    // Simuler l'envoi (dans un cas réel, faire un appel API ici)
    setTimeout(() => {
      // Déclencher un événement de succès
      container.dispatchEvent(new CustomEvent('contact-submit-success', {
        bubbles: true,
        detail: {
          formData: formData,
          form: form
        }
      }));

      // Afficher l'état de succès
      showSuccessState(container);
    }, 2000);
  }

  /**
   * Valide le formulaire
   * @param {HTMLFormElement} form - Élément form
   * @returns {{isValid: boolean, errors: Object}} Résultat de validation
   */
  function validateForm(form) {
    const errors = {};
    let isValid = true;

    // Récupérer tous les champs requis
    const requiredFields = form.querySelectorAll('[required]');

    requiredFields.forEach(field => {
      const fieldName = field.name || field.id;
      const fieldValue = field.value.trim();

      // Vérifier si le champ est vide
      if (!fieldValue) {
        errors[fieldName] = 'Ce champ est requis';
        isValid = false;
        return;
      }

      // Validation spécifique pour l'email
      if (field.type === 'email') {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(fieldValue)) {
          errors[fieldName] = 'Adresse email invalide';
          isValid = false;
        }
      }

      // Validation de longueur minimale pour le message
      if (field.tagName === 'TEXTAREA' && fieldValue.length < 10) {
        errors[fieldName] = 'Le message doit contenir au moins 10 caractères';
        isValid = false;
      }
    });

    return { isValid, errors };
  }

  /**
   * Affiche les erreurs de validation
   * @param {HTMLFormElement} form - Élément form
   * @param {Object} errors - Objet contenant les erreurs
   */
  function displayErrors(form, errors) {
    Object.keys(errors).forEach(fieldName => {
      const field = form.querySelector(`[name="${fieldName}"], #${fieldName}`);
      if (!field) return;

      const fieldContainer = field.closest('.seedit-contact-form__field');
      if (!fieldContainer) return;

      // Ajouter la classe d'erreur
      fieldContainer.classList.add('seedit-contact-form__field--error');

      // Créer et ajouter le message d'erreur
      const errorMsg = document.createElement('span');
      errorMsg.className = 'seedit-contact-form__error';
      errorMsg.textContent = errors[fieldName];
      errorMsg.setAttribute('role', 'alert');
      errorMsg.setAttribute('aria-live', 'polite');

      fieldContainer.appendChild(errorMsg);

      // Focus sur le premier champ en erreur
      if (Object.keys(errors)[0] === fieldName) {
        field.focus();
      }
    });
  }

  /**
   * Nettoie l'erreur d'un champ
   * @param {HTMLElement} fieldContainer - Conteneur du champ
   */
  function clearFieldError(fieldContainer) {
    fieldContainer.classList.remove('seedit-contact-form__field--error');

    const errorMsg = fieldContainer.querySelector('.seedit-contact-form__error');
    if (errorMsg) {
      errorMsg.remove();
    }
  }

  /**
   * Nettoie toutes les erreurs du formulaire
   * @param {HTMLFormElement} form - Élément form
   */
  function clearAllErrors(form) {
    const errorFields = form.querySelectorAll('.seedit-contact-form__field--error');
    errorFields.forEach(fieldContainer => {
      clearFieldError(fieldContainer);
    });
  }

  /**
   * Récupère les données du formulaire
   * @param {HTMLFormElement} form - Élément form
   * @returns {Object} Données du formulaire
   */
  function getFormData(form) {
    const formData = {};
    const fields = form.querySelectorAll('input, textarea, select');

    fields.forEach(field => {
      const fieldName = field.name || field.id;
      if (fieldName) {
        formData[fieldName] = field.value.trim();
      }
    });

    return formData;
  }

  /**
   * Affiche l'état de chargement
   * @param {HTMLFormElement} form - Élément form
   * @param {HTMLElement} container - Conteneur du formulaire
   */
  function showLoadingState(form, container) {
    // Ajouter la classe de chargement au formulaire
    form.classList.add('seedit-contact-form__form--loading');

    // Désactiver tous les champs
    const fields = form.querySelectorAll('input, textarea, button');
    fields.forEach(field => {
      field.disabled = true;
    });

    // Modifier le bouton de soumission
    const submitBtn = form.querySelector('.seedit-contact-form__submit');
    if (submitBtn) {
      submitBtn.classList.add('seedit-contact-form__submit--loading');

      // Sauvegarder le contenu original
      submitBtn.dataset.originalContent = submitBtn.innerHTML;

      // Afficher le spinner
      submitBtn.innerHTML = `
        <span class="seedit-contact-form__submit-spinner"></span>
        <span class="seedit-contact-form__submit-text">Envoi en cours...</span>
      `;
    }
  }

  /**
   * Affiche l'état de succès
   * @param {HTMLElement} container - Conteneur du formulaire
   */
  function showSuccessState(container) {
    // Ajouter la classe de succès
    container.classList.add('seedit-contact-form--success');

    // Animer l'apparition
    const successElement = container.querySelector('.seedit-contact-form__success');
    if (successElement) {
      successElement.style.opacity = '0';
      setTimeout(() => {
        successElement.style.transition = 'opacity 0.5s ease';
        successElement.style.opacity = '1';
      }, 50);
    }
  }

  /**
   * Réinitialise le formulaire
   * @param {HTMLFormElement} form - Élément form
   * @param {HTMLElement} container - Conteneur du formulaire
   */
  function resetForm(form, container) {
    // Retirer la classe de succès
    container.classList.remove('seedit-contact-form--success');

    // Réinitialiser le formulaire
    form.reset();

    // Retirer l'état de chargement
    form.classList.remove('seedit-contact-form__form--loading');

    // Réactiver tous les champs
    const fields = form.querySelectorAll('input, textarea, button');
    fields.forEach(field => {
      field.disabled = false;
    });

    // Réinitialiser le bouton de soumission
    const submitBtn = form.querySelector('.seedit-contact-form__submit');
    if (submitBtn) {
      submitBtn.classList.remove('seedit-contact-form__submit--loading');

      if (submitBtn.dataset.originalContent) {
        submitBtn.innerHTML = submitBtn.dataset.originalContent;
        delete submitBtn.dataset.originalContent;
      }
    }

    // Nettoyer les erreurs
    clearAllErrors(form);

    // Déclencher un événement de réinitialisation
    container.dispatchEvent(new CustomEvent('contact-reset', {
      bubbles: true,
      detail: {
        form: form
      }
    }));

    // Focus sur le premier champ
    const firstField = form.querySelector('input, textarea');
    if (firstField) {
      firstField.focus();
    }
  }

  /**
   * Initialisation de tous les formulaires au chargement du DOM
   */
  function init() {
    const forms = document.querySelectorAll('.seedit-contact-form__form');

    forms.forEach(form => {
      // Ne pas initialiser les formulaires de démonstration (avec erreurs pré-affichées)
      const isDemo = form.closest('.seedit-contact-demo__grid');
      if (!isDemo || form.id === 'contact-form-default') {
        initContactForm(form);
      }
    });
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer la fonction d'initialisation pour usage externe si nécessaire
  window.SeeditContactForm = {
    init: initContactForm
  };

})();
