/* ===================================
   Seed I.T - Input avec Effet Focus Animé
   Gestion des interactions et validation
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise un input avec effet focus
   * @param {HTMLElement} inputContainer - Conteneur de l'input
   */
  function initFocusInput(inputContainer) {
    if (!inputContainer) return;

    const input = inputContainer.querySelector('.seedit-focus-input__field');
    if (!input) return;

    // Déclencher un événement personnalisé au focus
    input.addEventListener('focus', () => {
      inputContainer.dispatchEvent(new CustomEvent('input-focus', {
        bubbles: true,
        detail: {
          input: input,
          value: input.value,
          container: inputContainer
        }
      }));
    });

    // Déclencher un événement personnalisé au blur
    input.addEventListener('blur', () => {
      inputContainer.dispatchEvent(new CustomEvent('input-blur', {
        bubbles: true,
        detail: {
          input: input,
          value: input.value,
          container: inputContainer
        }
      }));
    });

    // Déclencher un événement personnalisé au changement
    input.addEventListener('input', () => {
      inputContainer.dispatchEvent(new CustomEvent('input-change', {
        bubbles: true,
        detail: {
          input: input,
          value: input.value,
          container: inputContainer
        }
      }));
    });
  }

  /**
   * Initialise le formulaire d'exemple
   * @param {HTMLFormElement} form - Élément form
   */
  function initForm(form) {
    if (!form) return;

    form.addEventListener('submit', (e) => {
      e.preventDefault();

      // Récupérer les valeurs
      const formData = new FormData(form);
      const data = Object.fromEntries(formData.entries());

      // Validation simple
      const inputs = form.querySelectorAll('.seedit-focus-input__field[required]');
      let isValid = true;
      const errors = [];

      inputs.forEach(input => {
        const container = input.closest('.seedit-focus-input');
        if (!container) return;

        // Nettoyer les états précédents
        container.classList.remove('seedit-focus-input--error', 'seedit-focus-input--success');

        if (!input.value.trim()) {
          isValid = false;
          container.classList.add('seedit-focus-input--error');
          errors.push({
            field: input.id || input.name,
            message: 'Ce champ est requis'
          });
        } else if (input.type === 'email' && !isValidEmail(input.value)) {
          isValid = false;
          container.classList.add('seedit-focus-input--error');
          errors.push({
            field: input.id || input.name,
            message: 'Adresse email invalide'
          });
        } else {
          container.classList.add('seedit-focus-input--success');
        }
      });

      if (isValid) {
        // Déclencher un événement de succès
        form.dispatchEvent(new CustomEvent('form-submit-success', {
          bubbles: true,
          detail: {
            data: data,
            form: form
          }
        }));

        // Simuler une soumission réussie
        showSuccessMessage(form);

        // Réinitialiser après 3 secondes
        setTimeout(() => {
          form.reset();
          form.querySelectorAll('.seedit-focus-input').forEach(container => {
            container.classList.remove('seedit-focus-input--error', 'seedit-focus-input--success');
          });
        }, 3000);
      } else {
        // Déclencher un événement d'erreur
        form.dispatchEvent(new CustomEvent('form-submit-error', {
          bubbles: true,
          detail: {
            errors: errors,
            form: form
          }
        }));
      }
    });
  }

  /**
   * Valide une adresse email
   * @param {string} email - Adresse email à valider
   * @returns {boolean} True si valide
   */
  function isValidEmail(email) {
    const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return regex.test(email);
  }

  /**
   * Affiche un message de succès temporaire
   * @param {HTMLFormElement} form - Élément form
   */
  function showSuccessMessage(form) {
    const submitBtn = form.querySelector('.seedit-focus-input-form__submit');
    if (!submitBtn) return;

    const originalText = submitBtn.textContent;
    submitBtn.textContent = '✓ Formulaire envoyé !';
    submitBtn.style.background = 'linear-gradient(135deg, #10B981 0%, #059669 100%)';
    submitBtn.disabled = true;

    setTimeout(() => {
      submitBtn.textContent = originalText;
      submitBtn.style.background = '';
      submitBtn.disabled = false;
    }, 3000);
  }

  /**
   * Ajoute des validateurs personnalisés
   * @param {HTMLInputElement} input - Élément input
   * @param {Function} validator - Fonction de validation
   */
  function addValidator(input, validator) {
    if (!input || typeof validator !== 'function') return;

    const container = input.closest('.seedit-focus-input');
    if (!container) return;

    input.addEventListener('blur', () => {
      const isValid = validator(input.value);

      container.classList.remove('seedit-focus-input--error', 'seedit-focus-input--success');

      if (input.value.trim()) {
        if (isValid) {
          container.classList.add('seedit-focus-input--success');
        } else {
          container.classList.add('seedit-focus-input--error');
        }
      }
    });
  }

  /**
   * Réinitialise l'état de validation d'un input
   * @param {HTMLElement} inputContainer - Conteneur de l'input
   */
  function resetValidation(inputContainer) {
    if (!inputContainer) return;
    inputContainer.classList.remove('seedit-focus-input--error', 'seedit-focus-input--success');
  }

  /**
   * Définit l'état de validation d'un input
   * @param {HTMLElement} inputContainer - Conteneur de l'input
   * @param {string} state - 'success' ou 'error'
   */
  function setValidationState(inputContainer, state) {
    if (!inputContainer) return;

    inputContainer.classList.remove('seedit-focus-input--error', 'seedit-focus-input--success');

    if (state === 'success') {
      inputContainer.classList.add('seedit-focus-input--success');
    } else if (state === 'error') {
      inputContainer.classList.add('seedit-focus-input--error');
    }
  }

  /**
   * Initialisation de tous les inputs au chargement du DOM
   */
  function init() {
    // Initialiser tous les inputs avec effets focus
    const inputs = document.querySelectorAll('.seedit-focus-input');
    inputs.forEach(input => {
      initFocusInput(input);
    });

    // Initialiser le formulaire d'exemple
    const form = document.querySelector('.seedit-focus-input-form');
    if (form) {
      initForm(form);
    }

    // Exemple d'utilisation des événements personnalisés
    document.addEventListener('input-focus', (e) => {
      // console.log('Input focused:', e.detail);
    });

    document.addEventListener('input-blur', (e) => {
      // console.log('Input blurred:', e.detail);
    });

    document.addEventListener('input-change', (e) => {
      // console.log('Input changed:', e.detail);
    });
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer les fonctions d'initialisation pour usage externe si nécessaire
  window.SeeditFocusInput = {
    init: initFocusInput,
    addValidator: addValidator,
    resetValidation: resetValidation,
    setValidationState: setValidationState
  };

})();
