/* ===================================
   Seed I.T - Progress Ring Animé
   Gestion des animations et interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Calcule la circonférence du cercle (2 * π * r)
   * @param {number} radius - Rayon du cercle
   * @returns {number} Circonférence
   */
  function getCircleCircumference(radius) {
    return 2 * Math.PI * radius;
  }

  /**
   * Initialise un progress ring
   * @param {HTMLElement} ringElement - Élément .seedit-progress-ring
   */
  function initProgressRing(ringElement) {
    if (!ringElement) return;

    const fillCircle = ringElement.querySelector('.seedit-progress-ring__fill');
    const percentageText = ringElement.querySelector('.seedit-progress-ring__percentage');

    if (!fillCircle || !percentageText) return;

    // Récupérer le rayon depuis le SVG
    const radius = parseFloat(fillCircle.getAttribute('r'));
    const circumference = getCircleCircumference(radius);

    // Initialiser stroke-dasharray
    fillCircle.style.strokeDasharray = `${circumference} ${circumference}`;
    fillCircle.style.strokeDashoffset = circumference;

    // Récupérer la valeur de progression initiale
    const progress = parseInt(ringElement.getAttribute('data-progress') || '0', 10);

    // Mettre à jour le ring
    updateProgressRing(ringElement, progress);
  }

  /**
   * Met à jour la progression du ring
   * @param {HTMLElement} ringElement - Élément .seedit-progress-ring
   * @param {number} progress - Valeur de progression (0-100)
   */
  function updateProgressRing(ringElement, progress) {
    if (!ringElement) return;

    // Limiter entre 0 et 100
    progress = Math.max(0, Math.min(100, progress));

    const fillCircle = ringElement.querySelector('.seedit-progress-ring__fill');
    const percentageText = ringElement.querySelector('.seedit-progress-ring__percentage');

    if (!fillCircle || !percentageText) return;

    const radius = parseFloat(fillCircle.getAttribute('r'));
    const circumference = getCircleCircumference(radius);

    // Calculer le stroke-dashoffset basé sur le pourcentage
    const offset = circumference - (progress / 100) * circumference;
    fillCircle.style.strokeDashoffset = offset;

    // Mettre à jour le texte
    if (progress === 100) {
      percentageText.textContent = '100%';
      // Ajouter la classe complete si ce n'est pas déjà fait
      if (!ringElement.classList.contains('seedit-progress-ring--complete')) {
        ringElement.classList.add('seedit-progress-ring--complete');
      }
    } else {
      percentageText.textContent = `${Math.round(progress)}%`;
      ringElement.classList.remove('seedit-progress-ring--complete');
    }

    // Mettre à jour l'attribut data-progress
    ringElement.setAttribute('data-progress', progress);

    // Déclencher un événement personnalisé
    ringElement.dispatchEvent(new CustomEvent('progress-change', {
      bubbles: true,
      detail: {
        progress: progress,
        element: ringElement
      }
    }));

    // Si c'est terminé, déclencher un événement de complétion
    if (progress === 100) {
      ringElement.dispatchEvent(new CustomEvent('progress-complete', {
        bubbles: true,
        detail: {
          element: ringElement
        }
      }));
    }
  }

  /**
   * Anime la progression du ring de la valeur actuelle vers une cible
   * @param {HTMLElement} ringElement - Élément .seedit-progress-ring
   * @param {number} targetProgress - Valeur cible (0-100)
   * @param {number} duration - Durée de l'animation en ms (défaut: 1000)
   */
  function animateProgressRing(ringElement, targetProgress, duration = 1000) {
    if (!ringElement) return;

    const currentProgress = parseInt(ringElement.getAttribute('data-progress') || '0', 10);
    const difference = targetProgress - currentProgress;
    const startTime = performance.now();

    function animate(currentTime) {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);

      // Fonction d'easing (ease-out)
      const easedProgress = 1 - Math.pow(1 - progress, 3);
      const newProgress = currentProgress + (difference * easedProgress);

      updateProgressRing(ringElement, newProgress);

      if (progress < 1) {
        requestAnimationFrame(animate);
      }
    }

    requestAnimationFrame(animate);
  }

  /**
   * Réinitialise un progress ring à 0%
   * @param {HTMLElement} ringElement - Élément .seedit-progress-ring
   */
  function resetProgressRing(ringElement) {
    if (!ringElement) return;
    updateProgressRing(ringElement, 0);
  }

  /**
   * Initialise la démonstration interactive
   */
  function initInteractiveDemo() {
    const interactiveRing = document.getElementById('interactive-ring');
    const startBtn = document.getElementById('start-btn');
    const resetBtn = document.getElementById('reset-btn');

    if (!interactiveRing || !startBtn || !resetBtn) return;

    const sublabel = interactiveRing.querySelector('.seedit-progress-ring__sublabel');
    let intervalId = null;

    // Bouton démarrer
    startBtn.addEventListener('click', () => {
      // Annuler toute animation en cours
      if (intervalId) {
        clearInterval(intervalId);
      }

      // Réinitialiser si déjà à 100%
      const currentProgress = parseInt(interactiveRing.getAttribute('data-progress') || '0', 10);
      if (currentProgress === 100) {
        resetProgressRing(interactiveRing);
      }

      // Désactiver le bouton pendant l'animation
      startBtn.disabled = true;
      startBtn.textContent = 'En cours...';

      // Simuler une progression
      let progress = parseInt(interactiveRing.getAttribute('data-progress') || '0', 10);

      intervalId = setInterval(() => {
        progress += Math.random() * 8; // Incrément variable pour simuler un téléchargement réaliste

        if (progress >= 100) {
          progress = 100;
          clearInterval(intervalId);
          intervalId = null;

          // Réactiver le bouton
          startBtn.disabled = false;
          startBtn.textContent = 'Recommencer';

          if (sublabel) {
            sublabel.textContent = '100 / 100 Mo';
          }
        } else {
          if (sublabel) {
            sublabel.textContent = `${Math.round(progress)} / 100 Mo`;
          }
        }

        updateProgressRing(interactiveRing, progress);
      }, 150);
    });

    // Bouton réinitialiser
    resetBtn.addEventListener('click', () => {
      // Arrêter toute animation en cours
      if (intervalId) {
        clearInterval(intervalId);
        intervalId = null;
      }

      // Réinitialiser le ring
      resetProgressRing(interactiveRing);

      if (sublabel) {
        sublabel.textContent = '0 / 100 Mo';
      }

      // Réinitialiser le bouton démarrer
      startBtn.disabled = false;
      startBtn.textContent = 'Démarrer';
    });

    // Gestion du clavier pour accessibilité
    startBtn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        startBtn.click();
      }
    });

    resetBtn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        resetBtn.click();
      }
    });
  }

  /**
   * Initialisation de tous les progress rings au chargement du DOM
   */
  function init() {
    // Initialiser tous les progress rings existants
    const rings = document.querySelectorAll('.seedit-progress-ring');
    rings.forEach(ring => {
      initProgressRing(ring);
    });

    // Initialiser la démo interactive
    initInteractiveDemo();

    // Exemple d'utilisation des événements personnalisés
    document.addEventListener('progress-change', (e) => {
      // console.log('Progress changed:', e.detail.progress);
    });

    document.addEventListener('progress-complete', (e) => {
      // console.log('Progress complete!', e.detail.element);
    });
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer les fonctions pour usage externe si nécessaire
  window.SeeditProgressRing = {
    init: initProgressRing,
    update: updateProgressRing,
    animate: animateProgressRing,
    reset: resetProgressRing
  };

})();
