/* ===================================
   Seed I.T - Lignes de Séparation Décoratives
   Gestion des animations et interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise l'observation d'un divider pour l'animer au scroll
   * @param {HTMLElement} divider - Élément divider à observer
   */
  function initScrollAnimation(divider) {
    if (!divider) return;

    // Vérifier si le divider a déjà la classe animated
    if (!divider.classList.contains('seedit-divider--animated')) {
      return;
    }

    // Créer un observer pour détecter quand le divider entre dans le viewport
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            // Relancer l'animation en retirant et rajoutant la classe
            entry.target.style.animation = 'none';

            // Force un reflow
            void entry.target.offsetWidth;

            // Relancer l'animation
            entry.target.style.animation = '';

            // Optionnel : ne déclencher qu'une seule fois
            // observer.unobserve(entry.target);
          }
        });
      },
      {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
      }
    );

    observer.observe(divider);
  }

  /**
   * Crée un divider programmatiquement
   * @param {Object} options - Options de configuration du divider
   * @returns {HTMLElement} Élément divider créé
   */
  function createDivider(options = {}) {
    const {
      type = 'simple',
      text = '',
      icon = null,
      animated = false,
      thickness = 'normal',
      className = ''
    } = options;

    let divider;

    // Divider avec texte
    if (type === 'text' && text) {
      divider = document.createElement('div');
      divider.className = 'seedit-divider seedit-divider--text';

      const label = document.createElement('span');
      label.className = 'seedit-divider__label';
      label.textContent = text;

      divider.appendChild(label);
    }
    // Divider avec icône
    else if (type === 'icon' && icon) {
      divider = document.createElement('div');
      divider.className = 'seedit-divider seedit-divider--icon';

      const iconSpan = document.createElement('span');
      iconSpan.className = 'seedit-divider__icon';
      iconSpan.innerHTML = icon;

      divider.appendChild(iconSpan);
    }
    // Divider simple
    else {
      divider = document.createElement('hr');
      divider.className = 'seedit-divider';
    }

    // Ajouter les modificateurs de style
    if (type === 'gradient') {
      divider.classList.add('seedit-divider--gradient');
    } else if (type === 'glow') {
      divider.classList.add('seedit-divider--glow');
    } else if (type === 'dotted') {
      divider.classList.add('seedit-divider--dotted');
    } else if (type === 'dashed') {
      divider.classList.add('seedit-divider--dashed');
    }

    // Ajouter l'épaisseur
    if (thickness === 'thin') {
      divider.classList.add('seedit-divider--thin');
    } else if (thickness === 'thick') {
      divider.classList.add('seedit-divider--thick');
    } else if (thickness === 'extra-thick') {
      divider.classList.add('seedit-divider--extra-thick');
    }

    // Ajouter l'animation si demandée
    if (animated) {
      divider.classList.add('seedit-divider--animated');
    }

    // Ajouter une classe personnalisée si fournie
    if (className) {
      divider.classList.add(className);
    }

    return divider;
  }

  /**
   * Insère un divider après un élément
   * @param {HTMLElement} element - Élément de référence
   * @param {Object} options - Options du divider
   */
  function insertDividerAfter(element, options = {}) {
    if (!element || !element.parentNode) return;

    const divider = createDivider(options);
    element.parentNode.insertBefore(divider, element.nextSibling);

    // Initialiser l'animation au scroll si nécessaire
    if (options.animated) {
      initScrollAnimation(divider);
    }

    return divider;
  }

  /**
   * Insère un divider avant un élément
   * @param {HTMLElement} element - Élément de référence
   * @param {Object} options - Options du divider
   */
  function insertDividerBefore(element, options = {}) {
    if (!element || !element.parentNode) return;

    const divider = createDivider(options);
    element.parentNode.insertBefore(divider, element);

    // Initialiser l'animation au scroll si nécessaire
    if (options.animated) {
      initScrollAnimation(divider);
    }

    return divider;
  }

  /**
   * Initialisation de tous les dividers au chargement du DOM
   */
  function init() {
    // Initialiser les dividers animés avec scroll observation
    const animatedDividers = document.querySelectorAll('.seedit-divider--animated');
    animatedDividers.forEach(divider => {
      initScrollAnimation(divider);
    });

    // Événement personnalisé déclenché au chargement
    document.dispatchEvent(new CustomEvent('seedit-dividers-loaded', {
      bubbles: true,
      detail: {
        count: document.querySelectorAll('.seedit-divider').length
      }
    }));
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer les fonctions pour usage externe si nécessaire
  window.SeeditDivider = {
    create: createDivider,
    insertAfter: insertDividerAfter,
    insertBefore: insertDividerBefore,
    initScrollAnimation: initScrollAnimation
  };

})();
