/* ===================================
   Seed I.T - Breadcrumb Responsive
   Gestion du collapse et interactions
   =================================== */

(() => {
  'use strict';

  /**
   * Initialise le système de collapse pour breadcrumbs longs
   * @param {HTMLElement} breadcrumb - Élément .seedit-breadcrumb--collapsible
   */
  function initCollapsible(breadcrumb) {
    if (!breadcrumb) return;

    const list = breadcrumb.querySelector('.seedit-breadcrumb__list');
    if (!list) return;

    const items = Array.from(list.querySelectorAll('.seedit-breadcrumb__item'));
    const maxItems = parseInt(breadcrumb.getAttribute('data-max-items') || '4', 10);

    // Si moins d'éléments que le maximum, pas besoin de collapse
    if (items.length <= maxItems) return;

    // Conserver : premier élément, dernier élément, et quelques éléments avant le dernier
    const firstItem = items[0];
    const lastItem = items[items.length - 1];
    const keepBeforeLast = Math.max(1, maxItems - 2); // Garder au moins 1 élément avant le dernier

    // Masquer les éléments intermédiaires
    const collapsedItems = [];
    items.forEach((item, index) => {
      if (index === 0 || index >= items.length - keepBeforeLast - 1) {
        // Garder le premier et les derniers éléments
        return;
      }
      item.classList.add('seedit-breadcrumb__item--collapsed');
      collapsedItems.push(item);
    });

    // Créer l'élément "..." (ellipsis)
    if (collapsedItems.length > 0) {
      const ellipsisItem = document.createElement('li');
      ellipsisItem.className = 'seedit-breadcrumb__item';

      const ellipsisButton = document.createElement('button');
      ellipsisButton.className = 'seedit-breadcrumb__ellipsis';
      ellipsisButton.textContent = '...';
      ellipsisButton.setAttribute('aria-label', 'Afficher tous les éléments du fil d\'Ariane');
      ellipsisButton.setAttribute('type', 'button');

      ellipsisButton.addEventListener('click', () => {
        expandBreadcrumb(breadcrumb, collapsedItems, ellipsisItem);
      });

      ellipsisButton.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          expandBreadcrumb(breadcrumb, collapsedItems, ellipsisItem);
        }
      });

      ellipsisItem.appendChild(ellipsisButton);

      // Insérer l'ellipsis après le premier élément
      firstItem.after(ellipsisItem);
    }
  }

  /**
   * Expande un breadcrumb collapsed
   * @param {HTMLElement} breadcrumb - Élément breadcrumb
   * @param {Array} collapsedItems - Items masqués
   * @param {HTMLElement} ellipsisItem - Élément "..."
   */
  function expandBreadcrumb(breadcrumb, collapsedItems, ellipsisItem) {
    // Afficher tous les items masqués
    collapsedItems.forEach(item => {
      item.classList.remove('seedit-breadcrumb__item--collapsed');
    });

    // Retirer l'ellipsis
    if (ellipsisItem && ellipsisItem.parentNode) {
      ellipsisItem.remove();
    }

    // Déclencher un événement personnalisé
    breadcrumb.dispatchEvent(new CustomEvent('breadcrumb-expanded', {
      bubbles: true,
      detail: {
        breadcrumb: breadcrumb,
        expandedItems: collapsedItems
      }
    }));
  }

  /**
   * Crée un breadcrumb programmatiquement
   * @param {Array} items - Tableau d'objets {label, url, active}
   * @param {Object} options - Options de configuration
   * @returns {HTMLElement} Élément breadcrumb créé
   */
  function createBreadcrumb(items = [], options = {}) {
    const {
      separator = '>',
      withHomeIcon = false,
      variant = '', // 'boxed', 'pills', 'bordered', 'slash', etc.
      collapsible = false,
      maxItems = 4
    } = options;

    const nav = document.createElement('nav');
    nav.className = 'seedit-breadcrumb';
    nav.setAttribute('aria-label', 'Fil d\'Ariane');

    if (variant) {
      nav.classList.add(`seedit-breadcrumb--${variant}`);
    }

    if (collapsible) {
      nav.classList.add('seedit-breadcrumb--collapsible');
      nav.setAttribute('data-max-items', maxItems);
    }

    const list = document.createElement('ol');
    list.className = 'seedit-breadcrumb__list';

    items.forEach((item, index) => {
      const li = document.createElement('li');
      li.className = 'seedit-breadcrumb__item';

      if (item.active || index === items.length - 1) {
        li.classList.add('seedit-breadcrumb__item--active');
        const span = document.createElement('span');
        span.className = 'seedit-breadcrumb__current';
        span.setAttribute('aria-current', 'page');
        span.textContent = item.label;
        li.appendChild(span);
      } else {
        const link = document.createElement('a');
        link.className = 'seedit-breadcrumb__link';
        link.href = item.url || '#';

        // Ajouter l'icône maison pour le premier élément si demandé
        if (index === 0 && withHomeIcon) {
          const icon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
          icon.setAttribute('class', 'seedit-breadcrumb__home-icon');
          icon.setAttribute('width', '16');
          icon.setAttribute('height', '16');
          icon.setAttribute('viewBox', '0 0 16 16');
          icon.setAttribute('fill', 'none');

          const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
          path.setAttribute('d', 'M8 2L2 7V14H6V10H10V14H14V7L8 2Z');
          path.setAttribute('fill', 'currentColor');

          icon.appendChild(path);
          link.appendChild(icon);

          const srOnly = document.createElement('span');
          srOnly.className = 'seedit-breadcrumb__sr-only';
          srOnly.textContent = item.label;
          link.appendChild(srOnly);
        } else {
          link.textContent = item.label;
        }

        li.appendChild(link);
      }

      list.appendChild(li);
    });

    nav.appendChild(list);

    // Initialiser le collapse si nécessaire
    if (collapsible) {
      setTimeout(() => initCollapsible(nav), 0);
    }

    return nav;
  }

  /**
   * Initialisation de tous les breadcrumbs au chargement du DOM
   */
  function init() {
    // Initialiser tous les breadcrumbs collapsibles
    const collapsibleBreadcrumbs = document.querySelectorAll('.seedit-breadcrumb--collapsible');
    collapsibleBreadcrumbs.forEach(breadcrumb => {
      initCollapsible(breadcrumb);
    });

    // Événement personnalisé déclenché au chargement
    document.dispatchEvent(new CustomEvent('seedit-breadcrumbs-loaded', {
      bubbles: true,
      detail: {
        count: document.querySelectorAll('.seedit-breadcrumb').length
      }
    }));
  }

  // Lancement automatique au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // Exposer les fonctions pour usage externe si nécessaire
  window.SeeditBreadcrumb = {
    create: createBreadcrumb,
    initCollapsible: initCollapsible
  };

})();
