/* ============================================
   SEED I.T - Bouton avec effet d'électricité
   Génération procédurale d'arcs électriques en SVG
   ============================================ */

(() => {
  'use strict';

  // Configuration de l'effet électrique
  const CONFIG = {
    // Nombre d'arcs électriques par bouton
    numLightningBolts: 3,

    // Nombre de segments par arc (plus = plus détaillé)
    segmentsPerBolt: 8,

    // Amplitude maximale de déviation (en pourcentage)
    maxDeviation: 30,

    // Épaisseur des arcs
    strokeWidth: 2,

    // Couleur des arcs (sera surchargée par CSS variable)
    strokeColor: '#4DB6FF',

    // Opacité des arcs
    opacity: 0.8,

    // Durée de vie d'un arc (en ms)
    boltLifetime: 150,

    // Délai entre deux arcs (en ms)
    boltDelay: 50
  };

  /**
   * Génère un chemin SVG pour un arc électrique
   * @param {number} width - Largeur du conteneur
   * @param {number} height - Hauteur du conteneur
   * @returns {string} - Chemin SVG (attribut "d")
   */
  function generateLightningPath(width, height) {
    // Points de départ et d'arrivée aléatoires sur les bords
    const startX = Math.random() * width;
    const startY = 0;
    const endX = Math.random() * width;
    const endY = height;

    // Construction du chemin
    let path = `M ${startX} ${startY}`;

    // Génération des segments intermédiaires
    for (let i = 1; i < CONFIG.segmentsPerBolt; i++) {
      const progress = i / CONFIG.segmentsPerBolt;

      // Position linéaire
      let x = startX + (endX - startX) * progress;
      let y = startY + (endY - startY) * progress;

      // Déviation aléatoire
      const deviationX = (Math.random() - 0.5) * CONFIG.maxDeviation;
      const deviationY = (Math.random() - 0.5) * (CONFIG.maxDeviation / 2);

      x += deviationX;
      y += deviationY;

      // Ajout du point au chemin
      path += ` L ${x} ${y}`;
    }

    // Point final
    path += ` L ${endX} ${endY}`;

    return path;
  }

  /**
   * Crée un élément SVG path pour un arc électrique
   * @param {number} width - Largeur du conteneur
   * @param {number} height - Hauteur du conteneur
   * @param {string} color - Couleur de l'arc
   * @returns {SVGPathElement} - Élément SVG path
   */
  function createLightningBolt(width, height, color) {
    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');

    // Génération du chemin
    const d = generateLightningPath(width, height);
    path.setAttribute('d', d);

    // Style de l'arc
    path.setAttribute('stroke', color);
    path.setAttribute('stroke-width', CONFIG.strokeWidth);
    path.setAttribute('fill', 'none');
    path.setAttribute('opacity', CONFIG.opacity);
    path.setAttribute('stroke-linecap', 'round');

    // Effet de lueur (glow)
    path.setAttribute('filter', 'url(#seedit-glow-filter)');

    return path;
  }

  /**
   * Crée le filtre SVG de lueur (glow)
   * @returns {SVGFilterElement} - Élément SVG filter
   */
  function createGlowFilter() {
    const filter = document.createElementNS('http://www.w3.org/2000/svg', 'filter');
    filter.setAttribute('id', 'seedit-glow-filter');
    filter.setAttribute('x', '-50%');
    filter.setAttribute('y', '-50%');
    filter.setAttribute('width', '200%');
    filter.setAttribute('height', '200%');

    // Gaussian blur pour l'effet de lueur
    const feGaussianBlur = document.createElementNS('http://www.w3.org/2000/svg', 'feGaussianBlur');
    feGaussianBlur.setAttribute('stdDeviation', '3');
    feGaussianBlur.setAttribute('result', 'coloredBlur');

    // Merge pour combiner l'original et le flou
    const feMerge = document.createElementNS('http://www.w3.org/2000/svg', 'feMerge');

    const feMergeNode1 = document.createElementNS('http://www.w3.org/2000/svg', 'feMergeNode');
    feMergeNode1.setAttribute('in', 'coloredBlur');

    const feMergeNode2 = document.createElementNS('http://www.w3.org/2000/svg', 'feMergeNode');
    feMergeNode2.setAttribute('in', 'SourceGraphic');

    feMerge.appendChild(feMergeNode1);
    feMerge.appendChild(feMergeNode2);

    filter.appendChild(feGaussianBlur);
    filter.appendChild(feMerge);

    return filter;
  }

  /**
   * Anime les arcs électriques sur un bouton
   * @param {HTMLElement} button - Le bouton à animer
   */
  function animateLightning(button) {
    const svg = button.querySelector('.seedit-electric-btn__lightning');
    if (!svg) return;

    // Récupération de la couleur depuis CSS
    const computedStyle = getComputedStyle(button);
    const color = computedStyle.getPropertyValue('--seedit-electric-lightning').trim() || CONFIG.strokeColor;

    // Dimensions du SVG
    const rect = svg.getBoundingClientRect();
    const width = rect.width;
    const height = rect.height;

    // Nettoyage des arcs précédents
    while (svg.firstChild) {
      svg.removeChild(svg.firstChild);
    }

    // Ajout du filtre de lueur (une seule fois)
    const defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
    defs.appendChild(createGlowFilter());
    svg.appendChild(defs);

    // Génération des arcs avec délai
    for (let i = 0; i < CONFIG.numLightningBolts; i++) {
      setTimeout(() => {
        // Créer l'arc
        const bolt = createLightningBolt(width, height, color);
        svg.appendChild(bolt);

        // Animation d'apparition
        bolt.style.animation = `seedit-lightning-flash ${CONFIG.boltLifetime}ms ease-out`;

        // Supprimer l'arc après sa durée de vie
        setTimeout(() => {
          if (bolt.parentNode) {
            bolt.parentNode.removeChild(bolt);
          }
        }, CONFIG.boltLifetime);
      }, i * CONFIG.boltDelay);
    }
  }

  /**
   * Initialisation des boutons électriques
   */
  function initElectricButtons() {
    const buttons = document.querySelectorAll('.seedit-electric-btn');

    buttons.forEach((button) => {
      // Gestion du survol
      button.addEventListener('mouseenter', () => {
        animateLightning(button);
      });

      // Animation au focus (accessibilité)
      button.addEventListener('focus', () => {
        animateLightning(button);
      });

      // Gestion du clic avec effet visuel
      button.addEventListener('click', (e) => {
        // Animation de "pulse" au clic
        button.style.transform = 'scale(0.95)';
        setTimeout(() => {
          button.style.transform = '';
        }, 100);

        // Générer des arcs supplémentaires au clic
        animateLightning(button);

        console.log('Bouton électrique activé:', button.querySelector('.seedit-electric-btn__text')?.textContent);
      });
    });
  }

  // Ajout de l'animation CSS pour le flash des arcs
  const styleSheet = document.createElement('style');
  styleSheet.textContent = `
    @keyframes seedit-lightning-flash {
      0% {
        opacity: 0;
        stroke-width: 0;
      }
      20% {
        opacity: ${CONFIG.opacity};
        stroke-width: ${CONFIG.strokeWidth};
      }
      100% {
        opacity: 0;
        stroke-width: ${CONFIG.strokeWidth * 0.5};
      }
    }
  `;
  document.head.appendChild(styleSheet);

  // Initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initElectricButtons);
  } else {
    initElectricButtons();
  }

  console.log('Seed I.T - Bouton avec effet d\'électricité chargé ✓');
})();
