/* ============================================
   SEED I.T - Input avec effet de cassure de glace
   Génération procédurale de fissures et particules
   ============================================ */

(() => {
  'use strict';

  // Configuration de l'effet de glace
  const CONFIG = {
    // Fissures
    numMainCracks: 6,           // Nombre de fissures principales
    numBranches: 2,             // Branches par fissure
    crackLength: 120,           // Longueur moyenne des fissures (en %)
    crackVariation: 40,         // Variation de longueur (en %)
    crackSegments: 8,           // Segments par fissure
    branchProbability: 0.5,     // Probabilité de créer une branche (0-1)

    // Particules
    numParticles: 30,           // Nombre de particules par fissure
    particleSize: 2,            // Taille des particules
    particleSpeed: 2,           // Vitesse initiale
    particleGravity: 0.3,       // Gravité
    particleLifetime: 1200,     // Durée de vie (ms)

    // Animations
    crackAnimationDuration: 1500, // Durée totale incluant apparition + disparition (ms)
    crackDelay: 80,               // Délai entre chaque fissure (ms)
  };

  /**
   * Classe pour gérer les particules de glace
   */
  class IceParticle {
    constructor(x, y, canvas) {
      this.x = x;
      this.y = y;
      this.vx = (Math.random() - 0.5) * CONFIG.particleSpeed * 2;
      this.vy = -Math.random() * CONFIG.particleSpeed;
      this.size = Math.random() * CONFIG.particleSize + 1;
      this.opacity = 1;
      this.canvas = canvas;
      this.createdAt = Date.now();
    }

    update(deltaTime) {
      // Physique simple
      this.vy += CONFIG.particleGravity * deltaTime / 16;
      this.x += this.vx * deltaTime / 16;
      this.y += this.vy * deltaTime / 16;

      // Fade out progressif
      const age = Date.now() - this.createdAt;
      this.opacity = Math.max(0, 1 - age / CONFIG.particleLifetime);

      return this.opacity > 0;
    }

    draw(ctx) {
      ctx.save();
      ctx.globalAlpha = this.opacity;

      // Particule avec lueur
      const gradient = ctx.createRadialGradient(
        this.x, this.y, 0,
        this.x, this.y, this.size * 2
      );

      // Récupération de la couleur de particule depuis CSS
      const color = getComputedStyle(this.canvas.parentElement).getPropertyValue('--seedit-ice-particle').trim() || 'rgba(180, 220, 255, 0.8)';
      gradient.addColorStop(0, color);
      gradient.addColorStop(1, 'rgba(180, 220, 255, 0)');

      ctx.fillStyle = gradient;
      ctx.fillRect(this.x - this.size * 2, this.y - this.size * 2, this.size * 4, this.size * 4);

      ctx.restore();
    }
  }

  /**
   * Classe pour gérer l'animation des particules sur un canvas
   */
  class ParticleSystem {
    constructor(canvas) {
      this.canvas = canvas;
      this.ctx = canvas.getContext('2d');
      this.particles = [];
      this.animationId = null;
      this.lastTime = Date.now();
    }

    addParticle(x, y) {
      this.particles.push(new IceParticle(x, y, this.canvas));
    }

    start() {
      if (this.animationId) return;
      this.lastTime = Date.now();
      this.animate();
    }

    stop() {
      if (this.animationId) {
        cancelAnimationFrame(this.animationId);
        this.animationId = null;
      }
      this.particles = [];
      this.clear();
    }

    clear() {
      this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
    }

    animate() {
      const now = Date.now();
      const deltaTime = now - this.lastTime;
      this.lastTime = now;

      // Nettoyer le canvas
      this.clear();

      // Mettre à jour et dessiner les particules
      this.particles = this.particles.filter(particle => {
        const alive = particle.update(deltaTime);
        if (alive) {
          particle.draw(this.ctx);
        }
        return alive;
      });

      // Continuer l'animation s'il reste des particules
      if (this.particles.length > 0) {
        this.animationId = requestAnimationFrame(() => this.animate());
      } else {
        this.animationId = null;
      }
    }
  }

  /**
   * Génère un chemin SVG pour une fissure
   * @param {number} startX - Position X de départ (en pixels)
   * @param {number} startY - Position Y de départ (en pixels)
   * @param {number} angle - Angle de propagation (en radians)
   * @param {number} length - Longueur de la fissure (en pixels)
   * @param {number} segments - Nombre de segments
   * @returns {string} - Chemin SVG
   */
  function generateCrackPath(startX, startY, angle, length, segments) {
    let path = `M ${startX} ${startY}`;
    let currentX = startX;
    let currentY = startY;

    const segmentLength = length / segments;

    for (let i = 0; i < segments; i++) {
      // Déviation aléatoire de l'angle
      const angleDeviation = (Math.random() - 0.5) * Math.PI / 4;
      const currentAngle = angle + angleDeviation;

      // Variation de longueur
      const lengthVariation = 1 + (Math.random() - 0.5) * 0.4;
      const actualLength = segmentLength * lengthVariation;

      // Nouveau point (en pixels réels)
      currentX += Math.cos(currentAngle) * actualLength;
      currentY += Math.sin(currentAngle) * actualLength;

      path += ` L ${currentX} ${currentY}`;
    }

    return path;
  }

  /**
   * Génère une fissure principale avec ses branches
   * @param {number} width - Largeur du conteneur (en pixels)
   * @param {number} height - Hauteur du conteneur (en pixels)
   * @param {number} originX - Point d'origine X (en pixels)
   * @param {number} originY - Point d'origine Y (en pixels)
   * @returns {Array} - Tableau de chemins SVG
   */
  function generateCrackWithBranches(width, height, originX, originY) {
    const cracks = [];

    // Angle de base aléatoire
    const baseAngle = Math.random() * Math.PI * 2;

    // Longueur de base (en pixels) - utilise la diagonale comme référence
    const diagonal = Math.sqrt(width * width + height * height);
    const baseLengthPercent = CONFIG.crackLength + (Math.random() - 0.5) * CONFIG.crackVariation;
    const baseLength = (diagonal * baseLengthPercent) / 100;

    // Fissure principale
    const mainCrack = generateCrackPath(
      originX,
      originY,
      baseAngle,
      baseLength,
      CONFIG.crackSegments
    );
    cracks.push(mainCrack);

    // Branches
    for (let i = 0; i < CONFIG.numBranches; i++) {
      if (Math.random() < CONFIG.branchProbability) {
        // Position de départ de la branche (le long de la fissure principale)
        const branchStart = 0.3 + Math.random() * 0.4; // Entre 30% et 70% de la fissure principale
        const branchLength = baseLength * (0.4 + Math.random() * 0.3);

        // Angle de la branche (déviation par rapport à la fissure principale)
        const branchAngleOffset = (Math.random() - 0.5) * Math.PI / 2;
        const branchAngle = baseAngle + branchAngleOffset;

        // Point de départ de la branche (en pixels)
        const branchOriginX = originX + Math.cos(baseAngle) * baseLength * branchStart;
        const branchOriginY = originY + Math.sin(baseAngle) * baseLength * branchStart;

        const branch = generateCrackPath(
          branchOriginX,
          branchOriginY,
          branchAngle,
          branchLength,
          Math.floor(CONFIG.crackSegments * 0.6)
        );
        cracks.push(branch);
      }
    }

    return cracks;
  }

  /**
   * Crée un élément SVG path pour une fissure
   * @param {string} pathData - Données du chemin SVG
   * @param {number} delay - Délai d'animation (ms)
   * @returns {SVGPathElement}
   */
  function createCrackElement(pathData, delay) {
    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    path.setAttribute('d', pathData);
    path.setAttribute('class', 'seedit-ice-crack-path');

    // Animation avec délai
    setTimeout(() => {
      path.style.animation = `seedit-crack-appear ${CONFIG.crackAnimationDuration}ms ease-out forwards`;
    }, delay);

    return path;
  }

  /**
   * Génère les fissures de glace pour un input
   * @param {HTMLElement} wrapper - Conteneur de l'input
   * @param {number} focusX - Position X du focus (0-100, optionnel)
   * @param {number} focusY - Position Y du focus (0-100, optionnel)
   */
  function generateIceCracks(wrapper, focusX = 50, focusY = 50) {
    const svg = wrapper.querySelector('.seedit-ice-cracks');
    const canvas = wrapper.querySelector('.seedit-ice-particles');

    if (!svg || !canvas) return;

    // Dimensions
    const rect = svg.getBoundingClientRect();
    const width = rect.width;
    const height = rect.height;

    // Configuration du canvas
    canvas.width = width;
    canvas.height = height;

    // Système de particules
    const particleSystem = new ParticleSystem(canvas);

    // Nettoyage des fissures précédentes
    const existingPaths = svg.querySelectorAll('.seedit-ice-crack-path');
    existingPaths.forEach(path => path.remove());

    // Génération des fissures principales
    let totalDelay = 0;

    for (let i = 0; i < CONFIG.numMainCracks; i++) {
      // Conversion des pourcentages en pixels pour couvrir toute la surface
      const focusXPixels = (focusX / 100) * width;
      const focusYPixels = (focusY / 100) * height;

      // Dispersion large autour du point de focus (±60% de la taille)
      const originX = focusXPixels + (Math.random() - 0.5) * width * 1.2;
      const originY = focusYPixels + (Math.random() - 0.5) * height * 1.2;

      // Génération de la fissure avec branches
      const cracks = generateCrackWithBranches(width, height, originX, originY);

      // Création des éléments SVG
      cracks.forEach((crackPath, index) => {
        const delay = totalDelay + index * 50;
        const pathElement = createCrackElement(crackPath, delay);
        svg.appendChild(pathElement);

        // Génération de particules le long de la fissure
        setTimeout(() => {
          for (let j = 0; j < CONFIG.numParticles / CONFIG.numMainCracks; j++) {
            // Particules dispersées autour de l'origine (déjà en pixels)
            const particleX = originX + (Math.random() - 0.5) * 80;
            const particleY = originY + (Math.random() - 0.5) * 80;
            particleSystem.addParticle(particleX, particleY);
          }
          particleSystem.start();
        }, delay + 200);
      });

      totalDelay += CONFIG.crackDelay;
    }

    // Nettoyage après animation (les fissures disparaissent automatiquement)
    setTimeout(() => {
      const allPaths = svg.querySelectorAll('.seedit-ice-crack-path');
      allPaths.forEach(path => {
        if (path.parentNode) {
          path.parentNode.removeChild(path);
        }
      });
    }, totalDelay + CONFIG.crackAnimationDuration + 100);
  }

  /**
   * Nettoie les fissures et particules
   * @param {HTMLElement} wrapper - Conteneur de l'input
   */
  function clearIceCracks(wrapper) {
    const svg = wrapper.querySelector('.seedit-ice-cracks');
    const canvas = wrapper.querySelector('.seedit-ice-particles');

    if (svg) {
      const paths = svg.querySelectorAll('.seedit-ice-crack-path');
      paths.forEach(path => path.remove());
    }

    if (canvas) {
      const ctx = canvas.getContext('2d');
      ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
  }

  /**
   * Initialisation des inputs avec effet de glace
   */
  function initIceInputs() {
    const wrappers = document.querySelectorAll('.seedit-ice-input-wrapper');

    wrappers.forEach((wrapper) => {
      const input = wrapper.querySelector('.seedit-ice-input');
      if (!input) return;

      let hasCracked = false;

      // Gestion du focus
      input.addEventListener('focus', (e) => {
        if (!hasCracked) {
          // Calculer la position du focus (centre par défaut)
          const rect = wrapper.getBoundingClientRect();
          let focusX = 50;
          let focusY = 50;

          // Si on a un clic, utiliser la position du clic
          if (e.sourceCapabilities || e.detail) {
            // Position relative du clic
            focusX = 50; // Centre horizontal par défaut
            focusY = 50; // Centre vertical par défaut
          }

          generateIceCracks(wrapper, focusX, focusY);
          hasCracked = true;
        }
      });

      // Gestion du blur (quand l'input est vide, réinitialiser)
      input.addEventListener('blur', () => {
        if (!input.value) {
          setTimeout(() => {
            clearIceCracks(wrapper);
            hasCracked = false;
          }, 500);
        }
      });

      // Réinitialisation quand l'input est vidé
      input.addEventListener('input', () => {
        if (!input.value && !document.activeElement === input) {
          clearIceCracks(wrapper);
          hasCracked = false;
        }
      });
    });
  }

  /**
   * Gestion du redimensionnement de la fenêtre
   */
  function handleResize() {
    const wrappers = document.querySelectorAll('.seedit-ice-input-wrapper');
    wrappers.forEach((wrapper) => {
      const canvas = wrapper.querySelector('.seedit-ice-particles');
      const svg = wrapper.querySelector('.seedit-ice-cracks');

      if (canvas && svg) {
        const rect = svg.getBoundingClientRect();
        canvas.width = rect.width;
        canvas.height = rect.height;
      }
    });
  }

  // Throttle pour le resize
  let resizeTimeout;
  window.addEventListener('resize', () => {
    clearTimeout(resizeTimeout);
    resizeTimeout = setTimeout(handleResize, 250);
  });

  // Initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initIceInputs);
  } else {
    initIceInputs();
  }

  console.log('Seed I.T - Input avec effet de cassure de glace chargé ✓');
})();
