/* ============================================
   SEED I.T - Carte avec effet de rayons X
   Interactions et animations avancées
   ============================================ */

(() => {
  'use strict';

  /**
   * Initialisation des cartes avec effet rayons X
   */
  function initXRayCards() {
    const cards = document.querySelectorAll('.seedit-xray-card');

    cards.forEach((card) => {
      // État de la carte
      let isXRayActive = false;

      // Gestion du focus/hover
      card.addEventListener('mouseenter', () => {
        activateXRay(card);
        isXRayActive = true;
      });

      card.addEventListener('mouseleave', () => {
        deactivateXRay(card);
        isXRayActive = false;
      });

      // Accessibilité clavier
      card.addEventListener('keydown', (e) => {
        // Activation avec Enter ou Space
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();

          if (!isXRayActive) {
            activateXRay(card);
            isXRayActive = true;
          } else {
            deactivateXRay(card);
            isXRayActive = false;
          }
        }

        // Désactivation avec Escape
        if (e.key === 'Escape' && isXRayActive) {
          deactivateXRay(card);
          isXRayActive = false;
          card.blur();
        }
      });

      // Suivi de la souris pour effet de scan dynamique (optionnel)
      card.addEventListener('mousemove', (e) => {
        if (isXRayActive) {
          updateScanPosition(card, e);
        }
      });
    });
  }

  /**
   * Active l'effet rayons X sur une carte
   * @param {HTMLElement} card - La carte à activer
   */
  function activateXRay(card) {
    // Ajout de classe pour état actif (si besoin de style additionnel)
    card.classList.add('seedit-xray-card--active');

    // Animation des informations cachées (apparition progressive)
    const hiddenInfo = card.querySelector('.seedit-xray-card__hidden-info');
    if (hiddenInfo) {
      const items = hiddenInfo.querySelectorAll('.seedit-xray-info-item');
      items.forEach((item, index) => {
        item.style.animation = `seedit-xray-info-reveal 0.4s ease-out ${index * 0.1}s both`;
      });
    }
  }

  /**
   * Désactive l'effet rayons X sur une carte
   * @param {HTMLElement} card - La carte à désactiver
   */
  function deactivateXRay(card) {
    card.classList.remove('seedit-xray-card--active');

    // Réinitialisation des animations
    const hiddenInfo = card.querySelector('.seedit-xray-card__hidden-info');
    if (hiddenInfo) {
      const items = hiddenInfo.querySelectorAll('.seedit-xray-info-item');
      items.forEach((item) => {
        item.style.animation = '';
      });
    }
  }

  /**
   * Met à jour la position de la ligne de scan selon la souris
   * @param {HTMLElement} card - La carte
   * @param {MouseEvent} e - Événement de souris
   */
  function updateScanPosition(card, e) {
    const scan = card.querySelector('.seedit-xray-card__scan');
    if (!scan) return;

    // Récupération de la position relative dans la carte
    const rect = card.getBoundingClientRect();
    const relativeY = e.clientY - rect.top;
    const percentageY = (relativeY / rect.height) * 100;

    // Mise à jour de la position du scan (avec limite)
    const clampedY = Math.max(0, Math.min(100, percentageY));
    scan.style.top = `${clampedY}%`;
  }

  /**
   * Ajout de l'animation CSS pour les informations révélées
   */
  function addAnimationStyles() {
    const styleSheet = document.createElement('style');
    styleSheet.textContent = `
      @keyframes seedit-xray-info-reveal {
        0% {
          opacity: 0;
          transform: translateX(-20px);
        }
        100% {
          opacity: 1;
          transform: translateX(0);
        }
      }
    `;
    document.head.appendChild(styleSheet);
  }

  // Ajout des styles d'animation
  addAnimationStyles();

  // Initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initXRayCards);
  } else {
    initXRayCards();
  }

  console.log('Seed I.T - Carte avec effet de rayons X chargé');
})();
