// ====================================
// Menu Radial Contextuel - Seed I.T
// ====================================

(function() {
  'use strict';

  // Éléments du DOM
  const demoArea = document.getElementById('demoArea');
  const radialMenu = document.getElementById('radialMenu');
  const menuItems = document.querySelectorAll('.seedit-radial-menu__item');

  // Configuration
  const config = {
    itemCount: menuItems.length,
    radius: 120, // Distance du centre aux items (en pixels)
    animationDelay: 50, // Délai entre chaque item (en ms)
  };

  // État du menu
  let isMenuOpen = false;

  // ====================================
  // Fonction : Calculer les positions radiales
  // ====================================
  function calculateRadialPositions() {
    const angleStep = (2 * Math.PI) / config.itemCount; // Angle entre chaque item (en radians)
    const startAngle = -Math.PI / 2; // Commence en haut (12h)

    menuItems.forEach((item, index) => {
      const angle = startAngle + (angleStep * index);
      const x = Math.cos(angle) * config.radius;
      const y = Math.sin(angle) * config.radius;

      // Appliquer la position via transform
      // On garde translate(-50%, -50%) pour centrer l'item, puis on ajoute le décalage radial
      const baseTransform = 'translate(-50%, -50%)';
      item.style.transform = `${baseTransform} translate(${x}px, ${y}px) scale(0)`;

      // Stocker les coordonnées pour l'animation
      item.dataset.x = x;
      item.dataset.y = y;

      // Appliquer un délai d'animation progressif
      item.style.transitionDelay = `${index * config.animationDelay}ms`;
    });
  }

  // ====================================
  // Fonction : Afficher le menu
  // ====================================
  function showMenu(x, y) {
    // Positionner le menu au point de clic
    radialMenu.style.left = `${x}px`;
    radialMenu.style.top = `${y}px`;

    // Activer le menu
    radialMenu.classList.add('seedit-radial-menu--active');
    isMenuOpen = true;

    // Animer l'apparition des items
    setTimeout(() => {
      menuItems.forEach((item) => {
        const xPos = item.dataset.x;
        const yPos = item.dataset.y;
        item.style.transform = `translate(-50%, -50%) translate(${xPos}px, ${yPos}px) scale(1)`;
      });
    }, 10);
  }

  // ====================================
  // Fonction : Masquer le menu
  // ====================================
  function hideMenu() {
    if (!isMenuOpen) return;

    // Réinitialiser les items
    menuItems.forEach((item) => {
      item.style.transform = 'translate(-50%, -50%) scale(0)';
    });

    // Désactiver le menu après l'animation
    setTimeout(() => {
      radialMenu.classList.remove('seedit-radial-menu--active');
      isMenuOpen = false;
    }, 300);
  }

  // ====================================
  // Gestionnaire : Clic sur la zone de démo
  // ====================================
  demoArea.addEventListener('click', (e) => {
    e.preventDefault();
    e.stopPropagation();

    // Calculer la position du clic relative à la fenêtre
    const x = e.clientX;
    const y = e.clientY;

    // Fermer le menu s'il est déjà ouvert, sinon l'ouvrir
    if (isMenuOpen) {
      hideMenu();
    } else {
      showMenu(x, y);
    }
  });

  // ====================================
  // Gestionnaire : Clic sur un item du menu
  // ====================================
  menuItems.forEach((item) => {
    item.addEventListener('click', (e) => {
      e.stopPropagation();
      const action = item.dataset.action;

      // Simuler l'action (affichage dans la console)
      const actionNames = {
        copy: 'Copier',
        paste: 'Coller',
        edit: 'Modifier',
        delete: 'Supprimer',
        share: 'Partager',
        settings: 'Paramètres'
      };

      // Feedback visuel simple (optionnel)
      console.log(`Action déclenchée : ${actionNames[action] || action}`);

      // Fermer le menu après l'action
      hideMenu();
    });

    // Accessibilité : gérer l'activation avec Enter/Space
    item.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        item.click();
      }
    });
  });

  // ====================================
  // Gestionnaire : Clic en dehors du menu (fermeture)
  // ====================================
  document.addEventListener('click', (e) => {
    // Si le menu est ouvert et qu'on clique en dehors de la zone de démo
    if (isMenuOpen && !demoArea.contains(e.target) && !radialMenu.contains(e.target)) {
      hideMenu();
    }
  });

  // ====================================
  // Gestionnaire : Touche Escape (fermeture)
  // ====================================
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && isMenuOpen) {
      hideMenu();
    }
  });

  // ====================================
  // Gestionnaire : Prévenir le menu contextuel natif sur la zone de démo
  // ====================================
  demoArea.addEventListener('contextmenu', (e) => {
    e.preventDefault();
    // Optionnel : on pourrait aussi afficher le menu radial sur clic droit
    const x = e.clientX;
    const y = e.clientY;
    if (!isMenuOpen) {
      showMenu(x, y);
    }
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    // Calculer les positions initiales des items
    calculateRadialPositions();

    // Ajuster le rayon en fonction de la taille de l'écran
    function updateRadius() {
      const width = window.innerWidth;
      if (width <= 375) {
        config.radius = 85;
      } else if (width <= 768) {
        config.radius = 100;
      } else {
        config.radius = 120;
      }
      // Recalculer les positions si le menu n'est pas ouvert
      if (!isMenuOpen) {
        calculateRadialPositions();
      }
    }

    // Appeler au chargement
    updateRadius();

    // Recalculer au redimensionnement (avec debounce simple)
    let resizeTimeout;
    window.addEventListener('resize', () => {
      clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(() => {
        updateRadius();
      }, 150);
    });
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
