// ====================================
// Input avec Format Automatique - Seed I.T
// ====================================

(function() {
  'use strict';

  // Éléments du DOM
  const inputs = document.querySelectorAll('[data-format]');
  const validationStatus = document.getElementById('validationStatus');

  // ====================================
  // Fonctions de formatage
  // ====================================

  /**
   * Formate un numéro de téléphone français (10 chiffres)
   * Format : 06 12 34 56 78
   */
  function formatPhone(value) {
    // Ne garder que les chiffres
    const digits = value.replace(/\D/g, '');

    // Limiter à 10 chiffres
    const limited = digits.slice(0, 10);

    // Formater par groupes de 2
    const formatted = limited.match(/.{1,2}/g);

    return formatted ? formatted.join(' ') : '';
  }

  /**
   * Formate un numéro de carte bancaire (16 chiffres)
   * Format : 1234 5678 9012 3456
   */
  function formatCard(value) {
    // Ne garder que les chiffres
    const digits = value.replace(/\D/g, '');

    // Limiter à 16 chiffres
    const limited = digits.slice(0, 16);

    // Formater par groupes de 4
    const formatted = limited.match(/.{1,4}/g);

    return formatted ? formatted.join(' ') : '';
  }

  /**
   * Formate un code postal français (5 chiffres)
   * Format : 75001
   */
  function formatZip(value) {
    // Ne garder que les chiffres
    const digits = value.replace(/\D/g, '');

    // Limiter à 5 chiffres
    return digits.slice(0, 5);
  }

  // ====================================
  // Fonctions de validation
  // ====================================

  /**
   * Valide un numéro de téléphone (doit avoir 10 chiffres)
   */
  function validatePhone(value) {
    const digits = value.replace(/\D/g, '');
    return digits.length === 10;
  }

  /**
   * Valide un numéro de carte bancaire (doit avoir 16 chiffres)
   */
  function validateCard(value) {
    const digits = value.replace(/\D/g, '');
    return digits.length === 16;
  }

  /**
   * Valide un code postal (doit avoir 5 chiffres)
   */
  function validateZip(value) {
    const digits = value.replace(/\D/g, '');
    return digits.length === 5;
  }

  // ====================================
  // Gestion du formatage et de la validation
  // ====================================

  /**
   * Applique le formatage approprié selon le type d'input
   */
  function applyFormat(input, value) {
    const format = input.dataset.format;

    switch (format) {
      case 'phone':
        return formatPhone(value);
      case 'card':
        return formatCard(value);
      case 'zip':
        return formatZip(value);
      default:
        return value;
    }
  }

  /**
   * Valide la valeur selon le type d'input
   */
  function validateInput(input, value) {
    const format = input.dataset.format;

    switch (format) {
      case 'phone':
        return validatePhone(value);
      case 'card':
        return validateCard(value);
      case 'zip':
        return validateZip(value);
      default:
        return false;
    }
  }

  /**
   * Met à jour l'état visuel de validation d'un input
   */
  function updateValidationState(input) {
    const value = input.value;
    const isValid = validateInput(input, value);

    if (isValid) {
      input.classList.add('seedit-format-input__field--valid');
    } else {
      input.classList.remove('seedit-format-input__field--valid');
    }

    // Mettre à jour le statut global
    updateGlobalStatus();
  }

  /**
   * Met à jour l'indicateur de validation global
   */
  function updateGlobalStatus() {
    let allValid = true;
    let atLeastOneFilled = false;

    inputs.forEach(input => {
      if (input.value.length > 0) {
        atLeastOneFilled = true;
        if (!validateInput(input, input.value)) {
          allValid = false;
        }
      } else {
        allValid = false;
      }
    });

    // Afficher le statut seulement si tous les champs sont remplis et valides
    if (allValid && atLeastOneFilled) {
      validationStatus.classList.add('seedit-format-demo__status--visible');
    } else {
      validationStatus.classList.remove('seedit-format-demo__status--visible');
    }
  }

  // ====================================
  // Gestionnaire d'événements pour le formatage
  // ====================================

  /**
   * Gère l'événement d'input pour le formatage en temps réel
   */
  function handleInput(event) {
    const input = event.target;
    const cursorPosition = input.selectionStart;
    const oldValue = input.value;
    const oldLength = oldValue.length;

    // Appliquer le formatage
    const formattedValue = applyFormat(input, oldValue);

    // Mettre à jour la valeur
    input.value = formattedValue;

    // Calculer la nouvelle position du curseur
    // (pour éviter les sauts lors de l'ajout d'espaces automatiques)
    const newLength = formattedValue.length;
    const lengthDiff = newLength - oldLength;
    let newCursorPosition = cursorPosition + lengthDiff;

    // Ajuster si le curseur est sur un espace
    if (formattedValue[newCursorPosition - 1] === ' ') {
      newCursorPosition++;
    }

    // Restaurer la position du curseur
    input.setSelectionRange(newCursorPosition, newCursorPosition);

    // Mettre à jour l'état de validation
    updateValidationState(input);
  }

  /**
   * Gère l'événement de blur pour la validation finale
   */
  function handleBlur(event) {
    const input = event.target;
    updateValidationState(input);
  }

  /**
   * Gère l'événement de keydown pour autoriser uniquement les chiffres
   */
  function handleKeyDown(event) {
    const input = event.target;
    const key = event.key;

    // Autoriser : Backspace, Delete, Tab, Escape, Enter, touches de navigation
    const allowedKeys = [
      'Backspace', 'Delete', 'Tab', 'Escape', 'Enter',
      'ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown',
      'Home', 'End'
    ];

    // Autoriser Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
    if (event.ctrlKey || event.metaKey) {
      return;
    }

    // Si la touche est dans la liste autorisée, laisser passer
    if (allowedKeys.includes(key)) {
      return;
    }

    // Autoriser uniquement les chiffres (0-9)
    if (!/^\d$/.test(key)) {
      event.preventDefault();
    }
  }

  // ====================================
  // Initialisation
  // ====================================

  function init() {
    // Attacher les gestionnaires d'événements à chaque input
    inputs.forEach(input => {
      input.addEventListener('input', handleInput);
      input.addEventListener('blur', handleBlur);
      input.addEventListener('keydown', handleKeyDown);
    });

    // État initial du statut de validation
    updateGlobalStatus();
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
