// ====================================
// Widget Minuteur Pomodoro - Seed I.T
// ====================================

(function() {
  'use strict';

  // Éléments du DOM
  const timeDisplay = document.getElementById('timeDisplay');
  const progressRing = document.getElementById('progressRing');
  const playPauseBtn = document.getElementById('playPauseBtn');
  const playIcon = document.getElementById('playIcon');
  const pauseIcon = document.getElementById('pauseIcon');
  const playPauseText = document.getElementById('playPauseText');
  const resetBtn = document.getElementById('resetBtn');
  const skipBtn = document.getElementById('skipBtn');
  const sessionsCount = document.getElementById('sessionsCount');
  const phaseIndicator = document.getElementById('phaseIndicator');
  const phaseIcon = document.getElementById('phaseIcon');
  const phaseText = document.getElementById('phaseText');

  // Configuration des durées (en secondes)
  const DURATIONS = {
    work: 25 * 60,        // 25 minutes
    shortBreak: 5 * 60,   // 5 minutes
    longBreak: 15 * 60    // 15 minutes
  };

  // Configuration des phases
  const PHASES = {
    work: {
      name: 'Travail',
      icon: '💼',
      duration: DURATIONS.work,
      color: 'work'
    },
    shortBreak: {
      name: 'Pause courte',
      icon: '☕',
      duration: DURATIONS.shortBreak,
      color: 'short-break'
    },
    longBreak: {
      name: 'Pause longue',
      icon: '🌴',
      duration: DURATIONS.longBreak,
      color: 'long-break'
    }
  };

  // État du minuteur
  let currentPhase = 'work';
  let timeRemaining = PHASES[currentPhase].duration;
  let isRunning = false;
  let intervalId = null;
  let completedSessions = 0;

  // Constantes pour le cercle de progression
  const CIRCLE_CIRCUMFERENCE = 754; // 2 * PI * r (r = 120)

  // ====================================
  // Fonction : Formater le temps (MM:SS)
  // ====================================
  function formatTime(seconds) {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
  }

  // ====================================
  // Fonction : Mettre à jour l'affichage du temps
  // ====================================
  function updateDisplay() {
    timeDisplay.textContent = formatTime(timeRemaining);

    // Mettre à jour le cercle de progression
    const phase = PHASES[currentPhase];
    const progress = timeRemaining / phase.duration;
    const offset = CIRCLE_CIRCUMFERENCE * (1 - progress);
    progressRing.style.strokeDashoffset = offset;
  }

  // ====================================
  // Fonction : Mettre à jour l'indicateur de phase
  // ====================================
  function updatePhaseIndicator() {
    const phase = PHASES[currentPhase];

    // Mettre à jour le texte et l'icône
    phaseText.textContent = phase.name;
    phaseIcon.textContent = phase.icon;

    // Mettre à jour les classes de couleur
    phaseIndicator.className = 'seedit-pomodoro__phase';
    phaseIndicator.classList.add(`seedit-pomodoro__phase--${phase.color}`);

    // Mettre à jour la couleur du cercle
    progressRing.className = 'seedit-pomodoro__ring-progress';
    progressRing.classList.add(`seedit-pomodoro__ring-progress--${phase.color}`);
  }

  // ====================================
  // Fonction : Passer à la phase suivante
  // ====================================
  function nextPhase() {
    if (currentPhase === 'work') {
      completedSessions++;
      sessionsCount.textContent = completedSessions;

      // Après 4 sessions, pause longue
      if (completedSessions % 4 === 0) {
        currentPhase = 'longBreak';
      } else {
        currentPhase = 'shortBreak';
      }
    } else {
      // Après une pause, retour au travail
      currentPhase = 'work';
    }

    // Réinitialiser le temps
    timeRemaining = PHASES[currentPhase].duration;
    updatePhaseIndicator();
    updateDisplay();
  }

  // ====================================
  // Fonction : Démarrer/Mettre en pause
  // ====================================
  function togglePlayPause() {
    isRunning = !isRunning;

    if (isRunning) {
      // Démarrer le minuteur
      playIcon.style.display = 'none';
      pauseIcon.style.display = 'block';
      playPauseText.textContent = 'Pause';

      intervalId = setInterval(() => {
        timeRemaining--;

        if (timeRemaining < 0) {
          // Fin de la phase actuelle
          clearInterval(intervalId);
          isRunning = false;

          // Passer à la phase suivante
          nextPhase();

          // Redémarrer automatiquement (optionnel)
          // togglePlayPause();
        } else {
          updateDisplay();
        }
      }, 1000);
    } else {
      // Mettre en pause
      playIcon.style.display = 'block';
      pauseIcon.style.display = 'none';
      playPauseText.textContent = 'Démarrer';

      clearInterval(intervalId);
    }
  }

  // ====================================
  // Fonction : Réinitialiser
  // ====================================
  function reset() {
    // Arrêter le minuteur
    clearInterval(intervalId);
    isRunning = false;

    // Réinitialiser à la phase de travail
    currentPhase = 'work';
    timeRemaining = PHASES[currentPhase].duration;
    completedSessions = 0;

    // Mettre à jour l'affichage
    sessionsCount.textContent = completedSessions;
    updatePhaseIndicator();
    updateDisplay();

    // Réinitialiser les boutons
    playIcon.style.display = 'block';
    pauseIcon.style.display = 'none';
    playPauseText.textContent = 'Démarrer';
  }

  // ====================================
  // Fonction : Passer à la phase suivante (Skip)
  // ====================================
  function skip() {
    // Arrêter le minuteur actuel
    clearInterval(intervalId);
    isRunning = false;

    // Passer à la phase suivante
    nextPhase();

    // Réinitialiser les boutons
    playIcon.style.display = 'block';
    pauseIcon.style.display = 'none';
    playPauseText.textContent = 'Démarrer';
  }

  // ====================================
  // Gestionnaires d'événements
  // ====================================
  playPauseBtn.addEventListener('click', togglePlayPause);
  resetBtn.addEventListener('click', reset);
  skipBtn.addEventListener('click', skip);

  // Accessibilité : gestion du clavier
  [playPauseBtn, resetBtn, skipBtn].forEach(btn => {
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        btn.click();
      }
    });
  });

  // Gestion du raccourci clavier Espace pour play/pause
  document.addEventListener('keydown', (e) => {
    if (e.code === 'Space' && e.target === document.body) {
      e.preventDefault();
      togglePlayPause();
    }
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    // Initialiser l'affichage
    updatePhaseIndicator();
    updateDisplay();

    // Configurer le cercle de progression
    progressRing.style.strokeDasharray = CIRCLE_CIRCUMFERENCE;
    progressRing.style.strokeDashoffset = 0;
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  // ====================================
  // Nettoyage (optionnel)
  // ====================================
  window.addEventListener('beforeunload', () => {
    if (intervalId) {
      clearInterval(intervalId);
    }
  });

})();
