// ====================================
// Widget Tracker d'Objectifs - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Éléments du DOM
  // ====================================
  const goalsTracker = document.getElementById('goalsTracker');
  const goalsList = document.getElementById('goalsList');
  const goalsSummary = document.getElementById('goalsSummary');
  const globalProgress = document.getElementById('globalProgress');
  const globalProgressText = document.getElementById('globalProgressText');
  const emptyMessage = document.getElementById('emptyMessage');

  const addGoalBtn = document.getElementById('addGoalBtn');
  const emptyAddBtn = document.getElementById('emptyAddBtn');

  const goalModal = document.getElementById('goalModal');
  const modalOverlay = document.getElementById('modalOverlay');
  const modalClose = document.getElementById('modalClose');
  const modalCancel = document.getElementById('modalCancel');
  const goalForm = document.getElementById('goalForm');
  const goalTitle = document.getElementById('goalTitle');
  const goalTarget = document.getElementById('goalTarget');
  const goalCurrent = document.getElementById('goalCurrent');

  // ====================================
  // État de l'application
  // ====================================
  let goals = [];
  const STORAGE_KEY = 'seedit-goals-tracker';

  // ====================================
  // Fonction : Charger les objectifs depuis localStorage
  // ====================================
  function loadGoals() {
    try {
      const stored = localStorage.getItem(STORAGE_KEY);
      if (stored) {
        goals = JSON.parse(stored);
      }
    } catch (error) {
      console.error('Erreur lors du chargement des objectifs:', error);
      goals = [];
    }
  }

  // ====================================
  // Fonction : Sauvegarder les objectifs dans localStorage
  // ====================================
  function saveGoals() {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(goals));
    } catch (error) {
      console.error('Erreur lors de la sauvegarde des objectifs:', error);
    }
  }

  // ====================================
  // Fonction : Générer un ID unique
  // ====================================
  function generateId() {
    return `goal-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
  }

  // ====================================
  // Fonction : Calculer le pourcentage de progression
  // ====================================
  function calculateProgress(current, target) {
    if (target <= 0) return 0;
    const percentage = (current / target) * 100;
    return Math.min(100, Math.max(0, percentage));
  }

  // ====================================
  // Fonction : Mettre à jour l'affichage global
  // ====================================
  function updateGlobalStats() {
    const totalGoals = goals.length;
    const completedGoals = goals.filter(g => g.current >= g.target).length;

    // Mettre à jour le résumé
    goalsSummary.textContent = `${completedGoals} sur ${totalGoals} complétés`;

    // Calculer la progression globale
    let globalPercent = 0;
    if (totalGoals > 0) {
      globalPercent = (completedGoals / totalGoals) * 100;
    }

    // Mettre à jour la barre de progression
    globalProgress.style.width = `${globalPercent}%`;
    globalProgress.setAttribute('aria-valuenow', globalPercent.toFixed(0));
    globalProgressText.textContent = `${globalPercent.toFixed(0)}%`;

    // Afficher/masquer le message vide
    if (totalGoals === 0) {
      goalsTracker.classList.add('seedit-goals-tracker--empty');
    } else {
      goalsTracker.classList.remove('seedit-goals-tracker--empty');
    }
  }

  // ====================================
  // Fonction : Créer l'élément HTML d'un objectif
  // ====================================
  function createGoalElement(goal) {
    const li = document.createElement('li');
    li.className = 'seedit-goals-item';
    li.setAttribute('role', 'listitem');

    const isCompleted = goal.current >= goal.target;
    if (isCompleted) {
      li.classList.add('seedit-goals-item--completed');
    }

    const progress = calculateProgress(goal.current, goal.target);

    li.innerHTML = `
      <div class="seedit-goals-item__header">
        <h3 class="seedit-goals-item__title">${escapeHtml(goal.title)}</h3>
        <button
          class="seedit-goals-item__delete"
          data-id="${goal.id}"
          aria-label="Supprimer cet objectif"
          title="Supprimer"
        >
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="3 6 5 6 21 6"/>
            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
            <line x1="10" y1="11" x2="10" y2="17"/>
            <line x1="14" y1="11" x2="14" y2="17"/>
          </svg>
        </button>
      </div>

      <div class="seedit-goals-item__progress">
        <div class="seedit-goals-item__progress-bar">
          <div
            class="seedit-goals-item__progress-fill"
            style="width: ${progress}%"
            role="progressbar"
            aria-valuenow="${progress.toFixed(0)}"
            aria-valuemin="0"
            aria-valuemax="100"
            aria-label="Progression de l'objectif"
          ></div>
        </div>
        <span class="seedit-goals-item__progress-text">${goal.current}/${goal.target}</span>
      </div>

      <div class="seedit-goals-item__actions">
        <button
          class="seedit-goals-item__btn"
          data-id="${goal.id}"
          data-action="decrement"
          ${goal.current <= 0 ? 'disabled' : ''}
          aria-label="Décrémenter"
        >
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <line x1="5" y1="12" x2="19" y2="12"/>
          </svg>
          <span>-1</span>
        </button>
        <button
          class="seedit-goals-item__btn"
          data-id="${goal.id}"
          data-action="increment"
          ${goal.current >= goal.target ? 'disabled' : ''}
          aria-label="Incrémenter"
        >
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <line x1="12" y1="5" x2="12" y2="19"/>
            <line x1="5" y1="12" x2="19" y2="12"/>
          </svg>
          <span>+1</span>
        </button>
      </div>
    `;

    return li;
  }

  // ====================================
  // Fonction : Échapper le HTML (sécurité XSS)
  // ====================================
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // ====================================
  // Fonction : Afficher tous les objectifs
  // ====================================
  function renderGoals() {
    goalsList.innerHTML = '';

    goals.forEach(goal => {
      const goalElement = createGoalElement(goal);
      goalsList.appendChild(goalElement);
    });

    updateGlobalStats();
  }

  // ====================================
  // Fonction : Ajouter un objectif
  // ====================================
  function addGoal(title, target, current) {
    const newGoal = {
      id: generateId(),
      title: title.trim(),
      target: parseInt(target, 10),
      current: parseInt(current, 10)
    };

    goals.push(newGoal);
    saveGoals();
    renderGoals();
  }

  // ====================================
  // Fonction : Supprimer un objectif
  // ====================================
  function deleteGoal(id) {
    goals = goals.filter(g => g.id !== id);
    saveGoals();
    renderGoals();
  }

  // ====================================
  // Fonction : Incrémenter un objectif
  // ====================================
  function incrementGoal(id) {
    const goal = goals.find(g => g.id === id);
    if (goal && goal.current < goal.target) {
      goal.current++;
      saveGoals();
      renderGoals();
    }
  }

  // ====================================
  // Fonction : Décrémenter un objectif
  // ====================================
  function decrementGoal(id) {
    const goal = goals.find(g => g.id === id);
    if (goal && goal.current > 0) {
      goal.current--;
      saveGoals();
      renderGoals();
    }
  }

  // ====================================
  // Fonction : Ouvrir la modal
  // ====================================
  function openModal() {
    goalModal.classList.add('seedit-goals-modal--open');
    goalTitle.focus();

    // Empêcher le scroll du body
    document.body.style.overflow = 'hidden';
  }

  // ====================================
  // Fonction : Fermer la modal
  // ====================================
  function closeModal() {
    goalModal.classList.remove('seedit-goals-modal--open');
    goalForm.reset();

    // Réactiver le scroll du body
    document.body.style.overflow = '';
  }

  // ====================================
  // Gestionnaires d'événements
  // ====================================

  // Ouvrir la modal
  addGoalBtn.addEventListener('click', openModal);
  emptyAddBtn.addEventListener('click', openModal);

  // Fermer la modal
  modalClose.addEventListener('click', closeModal);
  modalCancel.addEventListener('click', closeModal);
  modalOverlay.addEventListener('click', closeModal);

  // Fermer avec Escape
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && goalModal.classList.contains('seedit-goals-modal--open')) {
      closeModal();
    }
  });

  // Soumettre le formulaire
  goalForm.addEventListener('submit', (e) => {
    e.preventDefault();

    const title = goalTitle.value.trim();
    const target = parseInt(goalTarget.value, 10);
    const current = parseInt(goalCurrent.value, 10);

    if (title && target > 0 && current >= 0) {
      addGoal(title, target, current);
      closeModal();
    }
  });

  // Délégation d'événements pour les actions sur les objectifs
  goalsList.addEventListener('click', (e) => {
    const deleteBtn = e.target.closest('.seedit-goals-item__delete');
    const actionBtn = e.target.closest('.seedit-goals-item__btn');

    if (deleteBtn) {
      const id = deleteBtn.getAttribute('data-id');
      if (confirm('Êtes-vous sûr de vouloir supprimer cet objectif ?')) {
        deleteGoal(id);
      }
    } else if (actionBtn && !actionBtn.disabled) {
      const id = actionBtn.getAttribute('data-id');
      const action = actionBtn.getAttribute('data-action');

      if (action === 'increment') {
        incrementGoal(id);
      } else if (action === 'decrement') {
        decrementGoal(id);
      }
    }
  });

  // Accessibilité : gestion clavier pour les boutons
  [addGoalBtn, emptyAddBtn, modalClose, modalCancel].forEach(btn => {
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        btn.click();
      }
    });
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    loadGoals();
    renderGoals();
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
