// ====================================
// Widget Liste de Tâches Avancé - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Éléments du DOM
  // ====================================
  const tasksWidget = document.getElementById('tasksWidget');
  const tasksList = document.getElementById('tasksList');
  const tasksCounter = document.getElementById('tasksCounter');
  const emptyMessage = document.getElementById('emptyMessage');
  const emptyMessageText = document.getElementById('emptyMessageText');

  const taskForm = document.getElementById('taskForm');
  const taskInput = document.getElementById('taskInput');
  const clearCompletedBtn = document.getElementById('clearCompletedBtn');

  const filterButtons = document.querySelectorAll('.seedit-tasks-widget__filter');

  // ====================================
  // État de l'application
  // ====================================
  let tasks = [];
  let currentFilter = 'all'; // 'all', 'active', 'completed'
  const STORAGE_KEY = 'seedit-tasks-widget';

  // ====================================
  // Fonction : Charger les tâches depuis localStorage
  // ====================================
  function loadTasks() {
    try {
      const stored = localStorage.getItem(STORAGE_KEY);
      if (stored) {
        tasks = JSON.parse(stored);
      }
    } catch (error) {
      console.error('Erreur lors du chargement des tâches:', error);
      tasks = [];
    }
  }

  // ====================================
  // Fonction : Sauvegarder les tâches dans localStorage
  // ====================================
  function saveTasks() {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(tasks));
    } catch (error) {
      console.error('Erreur lors de la sauvegarde des tâches:', error);
    }
  }

  // ====================================
  // Fonction : Générer un ID unique
  // ====================================
  function generateId() {
    return `task-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
  }

  // ====================================
  // Fonction : Échapper le HTML (sécurité XSS)
  // ====================================
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // ====================================
  // Fonction : Mettre à jour le compteur
  // ====================================
  function updateCounter() {
    const activeTasks = tasks.filter(t => !t.completed).length;
    tasksCounter.textContent = `${activeTasks} tâche(s) restante(s)`;

    // Activer/désactiver le bouton de nettoyage
    const hasCompleted = tasks.some(t => t.completed);
    clearCompletedBtn.disabled = !hasCompleted;
  }

  // ====================================
  // Fonction : Mettre à jour le message vide
  // ====================================
  function updateEmptyMessage() {
    const filteredTasks = getFilteredTasks();

    if (filteredTasks.length === 0) {
      tasksWidget.classList.add('seedit-tasks-widget--empty');

      // Message personnalisé selon le filtre
      if (currentFilter === 'active') {
        emptyMessageText.textContent = 'Aucune tâche active';
      } else if (currentFilter === 'completed') {
        emptyMessageText.textContent = 'Aucune tâche complétée';
      } else {
        emptyMessageText.textContent = 'Aucune tâche pour le moment';
      }
    } else {
      tasksWidget.classList.remove('seedit-tasks-widget--empty');
    }
  }

  // ====================================
  // Fonction : Obtenir les tâches filtrées
  // ====================================
  function getFilteredTasks() {
    if (currentFilter === 'active') {
      return tasks.filter(t => !t.completed);
    } else if (currentFilter === 'completed') {
      return tasks.filter(t => t.completed);
    }
    return tasks;
  }

  // ====================================
  // Fonction : Créer l'élément HTML d'une tâche
  // ====================================
  function createTaskElement(task) {
    const li = document.createElement('li');
    li.className = 'seedit-tasks-item';
    li.setAttribute('role', 'listitem');
    li.setAttribute('data-id', task.id);

    if (task.completed) {
      li.classList.add('seedit-tasks-item--completed');
    }

    // Appliquer le filtre
    const isVisible = shouldShowTask(task);
    if (!isVisible) {
      li.classList.add('seedit-tasks-item--hidden');
    }

    li.innerHTML = `
      <div class="seedit-tasks-item__checkbox" role="checkbox" aria-checked="${task.completed}" tabindex="0">
        <svg class="seedit-tasks-item__checkbox-icon" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
          <polyline points="20 6 9 17 4 12"/>
        </svg>
      </div>
      <span class="seedit-tasks-item__text" title="Double-cliquez pour éditer">${escapeHtml(task.text)}</span>
      <button
        class="seedit-tasks-item__delete"
        aria-label="Supprimer cette tâche"
        title="Supprimer"
      >
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <line x1="18" y1="6" x2="6" y2="18"/>
          <line x1="6" y1="6" x2="18" y2="18"/>
        </svg>
      </button>
    `;

    return li;
  }

  // ====================================
  // Fonction : Vérifier si une tâche doit être affichée
  // ====================================
  function shouldShowTask(task) {
    if (currentFilter === 'active') {
      return !task.completed;
    } else if (currentFilter === 'completed') {
      return task.completed;
    }
    return true;
  }

  // ====================================
  // Fonction : Afficher toutes les tâches
  // ====================================
  function renderTasks() {
    tasksList.innerHTML = '';

    tasks.forEach(task => {
      const taskElement = createTaskElement(task);
      tasksList.appendChild(taskElement);
    });

    updateCounter();
    updateEmptyMessage();
  }

  // ====================================
  // Fonction : Ajouter une tâche
  // ====================================
  function addTask(text) {
    const newTask = {
      id: generateId(),
      text: text.trim(),
      completed: false,
      createdAt: Date.now()
    };

    tasks.unshift(newTask);
    saveTasks();
    renderTasks();
  }

  // ====================================
  // Fonction : Basculer l'état de complétion
  // ====================================
  function toggleTask(id) {
    const task = tasks.find(t => t.id === id);
    if (task) {
      task.completed = !task.completed;
      saveTasks();
      renderTasks();
    }
  }

  // ====================================
  // Fonction : Supprimer une tâche
  // ====================================
  function deleteTask(id) {
    tasks = tasks.filter(t => t.id !== id);
    saveTasks();
    renderTasks();
  }

  // ====================================
  // Fonction : Éditer une tâche (inline)
  // ====================================
  function editTask(id, newText) {
    const task = tasks.find(t => t.id === id);
    if (task && newText.trim()) {
      task.text = newText.trim();
      saveTasks();
      renderTasks();
    }
  }

  // ====================================
  // Fonction : Entrer en mode édition
  // ====================================
  function enterEditMode(taskElement, taskId) {
    const task = tasks.find(t => t.id === taskId);
    if (!task) return;

    const textSpan = taskElement.querySelector('.seedit-tasks-item__text');
    const originalText = task.text;

    // Créer l'input d'édition
    const input = document.createElement('input');
    input.type = 'text';
    input.className = 'seedit-tasks-item__edit-input';
    input.value = originalText;
    input.maxLength = 100;

    // Remplacer le span par l'input
    textSpan.replaceWith(input);
    input.focus();
    input.select();

    // Fonction pour sauvegarder
    const saveEdit = () => {
      const newText = input.value.trim();
      if (newText && newText !== originalText) {
        editTask(taskId, newText);
      } else {
        renderTasks();
      }
    };

    // Fonction pour annuler
    const cancelEdit = () => {
      renderTasks();
    };

    // Événements
    input.addEventListener('blur', saveEdit);
    input.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        saveEdit();
      } else if (e.key === 'Escape') {
        cancelEdit();
      }
    });
  }

  // ====================================
  // Fonction : Supprimer toutes les tâches complétées
  // ====================================
  function clearCompleted() {
    const completedCount = tasks.filter(t => t.completed).length;

    if (completedCount === 0) return;

    if (confirm(`Supprimer ${completedCount} tâche(s) complétée(s) ?`)) {
      tasks = tasks.filter(t => !t.completed);
      saveTasks();
      renderTasks();
    }
  }

  // ====================================
  // Fonction : Changer le filtre
  // ====================================
  function setFilter(filter) {
    currentFilter = filter;

    // Mettre à jour les boutons
    filterButtons.forEach(btn => {
      const btnFilter = btn.getAttribute('data-filter');
      if (btnFilter === filter) {
        btn.classList.add('seedit-tasks-widget__filter--active');
        btn.setAttribute('aria-selected', 'true');
      } else {
        btn.classList.remove('seedit-tasks-widget__filter--active');
        btn.setAttribute('aria-selected', 'false');
      }
    });

    renderTasks();
  }

  // ====================================
  // Gestionnaires d'événements
  // ====================================

  // Soumettre le formulaire
  taskForm.addEventListener('submit', (e) => {
    e.preventDefault();

    const text = taskInput.value.trim();
    if (text) {
      addTask(text);
      taskInput.value = '';
    }
  });

  // Nettoyer les tâches complétées
  clearCompletedBtn.addEventListener('click', clearCompleted);

  // Gérer les filtres
  filterButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      const filter = btn.getAttribute('data-filter');
      setFilter(filter);
    });

    // Accessibilité clavier
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        btn.click();
      }
    });
  });

  // Délégation d'événements pour les actions sur les tâches
  tasksList.addEventListener('click', (e) => {
    const taskItem = e.target.closest('.seedit-tasks-item');
    if (!taskItem) return;

    const taskId = taskItem.getAttribute('data-id');

    // Clic sur la checkbox
    const checkbox = e.target.closest('.seedit-tasks-item__checkbox');
    if (checkbox) {
      toggleTask(taskId);
      return;
    }

    // Clic sur le bouton de suppression
    const deleteBtn = e.target.closest('.seedit-tasks-item__delete');
    if (deleteBtn) {
      deleteTask(taskId);
      return;
    }
  });

  // Double-clic pour éditer
  tasksList.addEventListener('dblclick', (e) => {
    const textSpan = e.target.closest('.seedit-tasks-item__text');
    if (textSpan) {
      const taskItem = textSpan.closest('.seedit-tasks-item');
      const taskId = taskItem.getAttribute('data-id');
      enterEditMode(taskItem, taskId);
    }
  });

  // Accessibilité clavier pour les checkboxes
  tasksList.addEventListener('keydown', (e) => {
    const checkbox = e.target.closest('.seedit-tasks-item__checkbox');
    if (checkbox && (e.key === 'Enter' || e.key === ' ')) {
      e.preventDefault();
      const taskItem = checkbox.closest('.seedit-tasks-item');
      const taskId = taskItem.getAttribute('data-id');
      toggleTask(taskId);
    }
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    loadTasks();
    renderTasks();
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
