// ====================================
// Widget Graphique de Performance - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Configuration
  // ====================================
  const UPDATE_INTERVAL = 2000; // 2 secondes
  const HISTORY_LENGTH = 20; // Points dans l'historique

  // Seuils pour les couleurs
  const THRESHOLDS = {
    cpu: { warning: 60, critical: 80 },
    ram: { warning: 70, critical: 85 },
    latency: { warning: 100, critical: 200 }, // ms
    throughput: { warning: 50, critical: 80 } // % de capacité max
  };

  // ====================================
  // Éléments du DOM
  // ====================================
  const metrics = {
    cpu: {
      valueEl: document.getElementById('cpuValue'),
      barEl: document.getElementById('cpuBar'),
      chartEl: document.getElementById('cpuChart'),
      metricEl: document.getElementById('metricCpu'),
      history: []
    },
    ram: {
      valueEl: document.getElementById('ramValue'),
      barEl: document.getElementById('ramBar'),
      chartEl: document.getElementById('ramChart'),
      metricEl: document.getElementById('metricRam'),
      history: []
    },
    latency: {
      valueEl: document.getElementById('latencyValue'),
      barEl: document.getElementById('latencyBar'),
      chartEl: document.getElementById('latencyChart'),
      metricEl: document.getElementById('metricLatency'),
      history: []
    },
    throughput: {
      valueEl: document.getElementById('throughputValue'),
      barEl: document.getElementById('throughputBar'),
      chartEl: document.getElementById('throughputChart'),
      metricEl: document.getElementById('metricThroughput'),
      history: []
    }
  };

  // ====================================
  // Fonction : Générer une valeur aléatoire réaliste
  // ====================================
  function generateValue(previousValue, min, max, volatility = 0.15) {
    if (previousValue === undefined) {
      return Math.random() * (max - min) + min;
    }

    const range = max - min;
    const change = (Math.random() - 0.5) * range * volatility;
    let newValue = previousValue + change;

    // Limiter aux bornes
    newValue = Math.max(min, Math.min(max, newValue));

    return newValue;
  }

  // ====================================
  // Fonction : Simuler des données de performance
  // ====================================
  function simulateData() {
    const lastCpu = metrics.cpu.history.length > 0
      ? metrics.cpu.history[metrics.cpu.history.length - 1]
      : undefined;

    const lastRam = metrics.ram.history.length > 0
      ? metrics.ram.history[metrics.ram.history.length - 1]
      : undefined;

    const lastLatency = metrics.latency.history.length > 0
      ? metrics.latency.history[metrics.latency.history.length - 1]
      : undefined;

    const lastThroughput = metrics.throughput.history.length > 0
      ? metrics.throughput.history[metrics.throughput.history.length - 1]
      : undefined;

    return {
      cpu: generateValue(lastCpu, 10, 95, 0.2),
      ram: generateValue(lastRam, 20, 90, 0.15),
      latency: generateValue(lastLatency, 10, 250, 0.3),
      throughput: generateValue(lastThroughput, 5, 100, 0.25)
    };
  }

  // ====================================
  // Fonction : Déterminer le niveau (good, warning, critical)
  // ====================================
  function getLevel(value, thresholds) {
    if (value >= thresholds.critical) {
      return 'critical';
    } else if (value >= thresholds.warning) {
      return 'warning';
    }
    return 'good';
  }

  // ====================================
  // Fonction : Mettre à jour une métrique
  // ====================================
  function updateMetric(metricName, value) {
    const metric = metrics[metricName];

    // Ajouter à l'historique
    metric.history.push(value);
    if (metric.history.length > HISTORY_LENGTH) {
      metric.history.shift();
    }

    // Déterminer le niveau
    let level;
    let displayValue;
    let barPercent;

    if (metricName === 'latency') {
      // Latence : valeur en ms
      level = getLevel(value, THRESHOLDS.latency);
      displayValue = `${Math.round(value)} ms`;
      barPercent = Math.min(100, (value / 300) * 100); // 300ms = 100%
    } else if (metricName === 'throughput') {
      // Débit : valeur en MB/s (simulé)
      level = getLevel(value, THRESHOLDS.throughput);
      displayValue = `${value.toFixed(1)} MB/s`;
      barPercent = value;
    } else {
      // CPU et RAM : pourcentages
      level = getLevel(value, THRESHOLDS[metricName]);
      displayValue = `${Math.round(value)}%`;
      barPercent = value;
    }

    // Mettre à jour la valeur affichée
    metric.valueEl.textContent = displayValue;

    // Mettre à jour la barre de progression
    metric.barEl.style.width = `${barPercent}%`;
    metric.barEl.setAttribute('aria-valuenow', Math.round(barPercent));

    // Mettre à jour les classes de couleur
    metric.barEl.className = 'seedit-perf-metric__bar-fill';
    metric.barEl.classList.add(`seedit-perf-metric__bar-fill--${level}`);

    metric.metricEl.className = 'seedit-perf-metric';
    metric.metricEl.classList.add(`seedit-perf-metric--${level}`);

    // Mettre à jour le graphique sparkline
    updateChart(metric);
  }

  // ====================================
  // Fonction : Mettre à jour le graphique sparkline
  // ====================================
  function updateChart(metric) {
    const history = metric.history;
    if (history.length === 0) return;

    const chartEl = metric.chartEl;
    const polyline = chartEl.querySelector('.seedit-perf-metric__chart-line');

    // Calculer les coordonnées du polyline
    const width = 100;
    const height = 30;
    const step = width / (HISTORY_LENGTH - 1);

    // Normaliser les valeurs (0-100 scale)
    const maxValue = Math.max(...history, 1);
    const minValue = Math.min(...history);
    const range = maxValue - minValue || 1;

    const points = history.map((value, index) => {
      const x = index * step;
      const normalized = (value - minValue) / range;
      const y = height - (normalized * height);
      return `${x.toFixed(2)},${y.toFixed(2)}`;
    }).join(' ');

    polyline.setAttribute('points', points);
  }

  // ====================================
  // Fonction : Mettre à jour toutes les métriques
  // ====================================
  function updateAllMetrics() {
    const data = simulateData();

    updateMetric('cpu', data.cpu);
    updateMetric('ram', data.ram);
    updateMetric('latency', data.latency);
    updateMetric('throughput', data.throughput);
  }

  // ====================================
  // Fonction : Démarrer la mise à jour automatique
  // ====================================
  let updateIntervalId = null;

  function startUpdates() {
    // Première mise à jour immédiate
    updateAllMetrics();

    // Mises à jour périodiques
    updateIntervalId = setInterval(updateAllMetrics, UPDATE_INTERVAL);
  }

  // ====================================
  // Fonction : Arrêter les mises à jour
  // ====================================
  function stopUpdates() {
    if (updateIntervalId) {
      clearInterval(updateIntervalId);
      updateIntervalId = null;
    }
  }

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    // Initialiser l'historique avec des valeurs de base
    for (let i = 0; i < HISTORY_LENGTH; i++) {
      const data = simulateData();
      metrics.cpu.history.push(data.cpu);
      metrics.ram.history.push(data.ram);
      metrics.latency.history.push(data.latency);
      metrics.throughput.history.push(data.throughput);
    }

    // Démarrer les mises à jour
    startUpdates();
  }

  // ====================================
  // Nettoyage avant déchargement de la page
  // ====================================
  window.addEventListener('beforeunload', () => {
    stopUpdates();
  });

  // ====================================
  // Gestion de la visibilité de la page (économie de ressources)
  // ====================================
  document.addEventListener('visibilitychange', () => {
    if (document.hidden) {
      stopUpdates();
    } else {
      startUpdates();
    }
  });

  // ====================================
  // Lancer l'initialisation au chargement du DOM
  // ====================================
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
