// ====================================
// Widget de Sondage Interactif - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Configuration du sondage
  // ====================================
  const POLL_DATA = {
    question: "Quel langage de programmation préférez-vous ?",
    options: [
      { id: 'js', label: 'JavaScript / TypeScript', votes: 42 },
      { id: 'python', label: 'Python', votes: 38 },
      { id: 'rust', label: 'Rust', votes: 15 },
      { id: 'go', label: 'Go', votes: 23 },
      { id: 'java', label: 'Java', votes: 18 }
    ]
  };

  const STORAGE_KEY = 'seedit-poll-vote';

  // ====================================
  // Éléments du DOM
  // ====================================
  const pollQuestion = document.getElementById('pollQuestion');
  const votesCount = document.getElementById('votesCount');
  const pollForm = document.getElementById('pollForm');
  const pollOptions = document.getElementById('pollOptions');
  const voteBtn = document.getElementById('voteBtn');
  const viewResultsBtn = document.getElementById('viewResultsBtn');
  const pollResults = document.getElementById('pollResults');
  const resultsList = document.getElementById('resultsList');
  const thanksMessage = document.getElementById('thanksMessage');
  const resetBtn = document.getElementById('resetBtn');

  // ====================================
  // État de l'application
  // ====================================
  let selectedOption = null;
  let hasVoted = false;
  let userVote = null;

  // ====================================
  // Fonction : Charger les données de vote depuis localStorage
  // ====================================
  function loadVoteData() {
    try {
      const stored = localStorage.getItem(STORAGE_KEY);
      if (stored) {
        const data = JSON.parse(stored);
        hasVoted = data.hasVoted || false;
        userVote = data.userVote || null;
      }
    } catch (error) {
      console.error('Erreur lors du chargement des données de vote:', error);
    }
  }

  // ====================================
  // Fonction : Sauvegarder les données de vote dans localStorage
  // ====================================
  function saveVoteData() {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify({
        hasVoted,
        userVote
      }));
    } catch (error) {
      console.error('Erreur lors de la sauvegarde des données de vote:', error);
    }
  }

  // ====================================
  // Fonction : Calculer le total de votes
  // ====================================
  function getTotalVotes() {
    return POLL_DATA.options.reduce((sum, option) => sum + option.votes, 0);
  }

  // ====================================
  // Fonction : Mettre à jour le compteur de votes
  // ====================================
  function updateVotesCount() {
    const total = getTotalVotes();
    votesCount.textContent = `${total} vote${total > 1 ? 's' : ''}`;
  }

  // ====================================
  // Fonction : Créer les options de vote
  // ====================================
  function renderOptions() {
    pollOptions.innerHTML = '';

    POLL_DATA.options.forEach(option => {
      const optionDiv = document.createElement('div');
      optionDiv.className = 'seedit-poll-option';
      optionDiv.setAttribute('role', 'radio');
      optionDiv.setAttribute('aria-checked', 'false');
      optionDiv.setAttribute('tabindex', '0');
      optionDiv.setAttribute('data-option-id', option.id);

      optionDiv.innerHTML = `
        <input
          type="radio"
          name="poll-option"
          id="option-${option.id}"
          value="${option.id}"
          class="seedit-poll-option__input"
        />
        <div class="seedit-poll-option__radio" aria-hidden="true"></div>
        <label for="option-${option.id}" class="seedit-poll-option__label">${escapeHtml(option.label)}</label>
      `;

      // Événements
      optionDiv.addEventListener('click', () => selectOption(option.id));
      optionDiv.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          selectOption(option.id);
        }
      });

      pollOptions.appendChild(optionDiv);
    });
  }

  // ====================================
  // Fonction : Échapper le HTML (sécurité XSS)
  // ====================================
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // ====================================
  // Fonction : Sélectionner une option
  // ====================================
  function selectOption(optionId) {
    selectedOption = optionId;

    // Mettre à jour l'UI
    const allOptions = pollOptions.querySelectorAll('.seedit-poll-option');
    allOptions.forEach(opt => {
      const id = opt.getAttribute('data-option-id');
      if (id === optionId) {
        opt.classList.add('seedit-poll-option--selected');
        opt.setAttribute('aria-checked', 'true');
        opt.querySelector('input').checked = true;
      } else {
        opt.classList.remove('seedit-poll-option--selected');
        opt.setAttribute('aria-checked', 'false');
        opt.querySelector('input').checked = false;
      }
    });

    // Activer le bouton de vote
    voteBtn.disabled = false;
  }

  // ====================================
  // Fonction : Soumettre le vote
  // ====================================
  function submitVote() {
    if (!selectedOption) return;

    // Enregistrer le vote
    const option = POLL_DATA.options.find(opt => opt.id === selectedOption);
    if (option) {
      option.votes++;
    }

    hasVoted = true;
    userVote = selectedOption;
    saveVoteData();

    // Afficher les résultats
    showResults();
  }

  // ====================================
  // Fonction : Afficher les résultats
  // ====================================
  function showResults() {
    // Masquer le formulaire
    pollForm.style.display = 'none';

    // Afficher les résultats
    pollResults.style.display = 'block';

    // Mettre à jour le message de remerciement
    if (hasVoted && userVote) {
      thanksMessage.textContent = 'Merci d\'avoir participé !';
    } else {
      thanksMessage.textContent = 'Aperçu des résultats';
    }

    // Générer les résultats
    renderResults();

    // Mettre à jour le compteur
    updateVotesCount();
  }

  // ====================================
  // Fonction : Afficher les options de vote
  // ====================================
  function showVoteForm() {
    pollForm.style.display = 'flex';
    pollResults.style.display = 'none';
  }

  // ====================================
  // Fonction : Générer l'affichage des résultats
  // ====================================
  function renderResults() {
    resultsList.innerHTML = '';

    const totalVotes = getTotalVotes();

    // Trier par nombre de votes (décroissant)
    const sortedOptions = [...POLL_DATA.options].sort((a, b) => b.votes - a.votes);
    const maxVotes = sortedOptions[0].votes;

    sortedOptions.forEach(option => {
      const percentage = totalVotes > 0 ? (option.votes / totalVotes) * 100 : 0;
      const isWinner = option.votes === maxVotes && maxVotes > 0;
      const isUserChoice = option.id === userVote;

      const resultDiv = document.createElement('div');
      resultDiv.className = 'seedit-poll-result';

      if (isWinner) {
        resultDiv.classList.add('seedit-poll-result--winner');
      }
      if (isUserChoice) {
        resultDiv.classList.add('seedit-poll-result--selected');
      }

      resultDiv.innerHTML = `
        <div class="seedit-poll-result__bar" style="width: ${percentage}%"></div>
        <div class="seedit-poll-result__content">
          <div class="seedit-poll-result__label">
            ${escapeHtml(option.label)}
            ${isWinner ? '<span class="seedit-poll-result__badge">Gagnant</span>' : ''}
            ${isUserChoice ? '<span class="seedit-poll-result__badge" style="background: var(--seedit-primary-light)">Votre choix</span>' : ''}
          </div>
          <div class="seedit-poll-result__stats">
            <span class="seedit-poll-result__percentage">${percentage.toFixed(1)}%</span>
            <span class="seedit-poll-result__votes">${option.votes} vote${option.votes > 1 ? 's' : ''}</span>
          </div>
        </div>
      `;

      resultsList.appendChild(resultDiv);
    });

    // Animation des barres (légèrement décalée)
    setTimeout(() => {
      const bars = resultsList.querySelectorAll('.seedit-poll-result__bar');
      bars.forEach((bar, index) => {
        setTimeout(() => {
          const width = bar.style.width;
          bar.style.width = '0%';
          requestAnimationFrame(() => {
            bar.style.width = width;
          });
        }, index * 100);
      });
    }, 50);
  }

  // ====================================
  // Fonction : Réinitialiser le sondage
  // ====================================
  function resetPoll() {
    // Réinitialiser l'état
    hasVoted = false;
    userVote = null;
    selectedOption = null;

    // Effacer le localStorage
    try {
      localStorage.removeItem(STORAGE_KEY);
    } catch (error) {
      console.error('Erreur lors de la suppression des données de vote:', error);
    }

    // Afficher le formulaire
    showVoteForm();

    // Re-générer les options
    renderOptions();

    // Désactiver le bouton de vote
    voteBtn.disabled = true;

    // Mettre à jour le compteur
    updateVotesCount();
  }

  // ====================================
  // Gestionnaires d'événements
  // ====================================

  // Soumettre le formulaire
  pollForm.addEventListener('submit', (e) => {
    e.preventDefault();
    if (selectedOption) {
      submitVote();
    }
  });

  // Voir les résultats sans voter
  viewResultsBtn.addEventListener('click', () => {
    showResults();
  });

  // Réinitialiser
  resetBtn.addEventListener('click', () => {
    if (confirm('Réinitialiser le sondage et pouvoir voter à nouveau ?')) {
      resetPoll();
    }
  });

  // Accessibilité clavier pour les boutons
  [voteBtn, viewResultsBtn, resetBtn].forEach(btn => {
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        btn.click();
      }
    });
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    // Charger les données de vote
    loadVoteData();

    // Afficher la question
    pollQuestion.textContent = POLL_DATA.question;

    // Mettre à jour le compteur
    updateVotesCount();

    // Si l'utilisateur a déjà voté, afficher les résultats
    if (hasVoted) {
      showResults();
    } else {
      // Sinon, afficher le formulaire
      renderOptions();
      voteBtn.disabled = true;
    }
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
