// ====================================
// Widget Calendrier Événements - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Données d'exemple d'événements
  // ====================================
  const EVENTS = [
    {
      id: 1,
      title: 'Réunion équipe',
      date: '2024-11-24',
      time: '10:00',
      category: 'meeting'
    },
    {
      id: 2,
      title: 'Présentation client',
      date: '2024-11-24',
      time: '14:30',
      category: 'work'
    },
    {
      id: 3,
      title: 'Date limite projet X',
      date: '2024-11-25',
      time: '17:00',
      category: 'deadline'
    },
    {
      id: 4,
      title: 'Rendez-vous médecin',
      date: '2024-11-26',
      time: '09:00',
      category: 'personal'
    },
    {
      id: 5,
      title: 'Call hebdomadaire',
      date: '2024-11-27',
      time: '11:00',
      category: 'meeting'
    },
    {
      id: 6,
      title: 'Formation JavaScript',
      date: '2024-11-28',
      time: '13:00',
      category: 'work'
    },
    {
      id: 7,
      title: 'Déjeuner team building',
      date: '2024-11-29',
      time: '12:30',
      category: 'personal'
    },
    {
      id: 8,
      title: 'Sprint planning',
      date: '2024-12-02',
      time: '09:30',
      category: 'meeting'
    }
  ];

  const CATEGORY_LABELS = {
    work: 'Travail',
    personal: 'Personnel',
    meeting: 'Réunion',
    deadline: 'Échéance'
  };

  const MONTH_NAMES = [
    'Janvier', 'Février', 'Mars', 'Avril', 'Mai', 'Juin',
    'Juillet', 'Août', 'Septembre', 'Octobre', 'Novembre', 'Décembre'
  ];

  // ====================================
  // Éléments du DOM
  // ====================================
  const calendarTitle = document.getElementById('calendarTitle');
  const calendarDays = document.getElementById('calendarDays');
  const prevMonthBtn = document.getElementById('prevMonthBtn');
  const nextMonthBtn = document.getElementById('nextMonthBtn');
  const eventsPanel = document.getElementById('eventsPanel');
  const eventsTitle = document.getElementById('eventsTitle');
  const eventsList = document.getElementById('eventsList');

  // ====================================
  // État de l'application
  // ====================================
  let currentDate = new Date();
  let selectedDate = null;
  const today = new Date();
  today.setHours(0, 0, 0, 0);

  // ====================================
  // Fonction : Obtenir le premier jour du mois (jour de la semaine)
  // ====================================
  function getFirstDayOfMonth(year, month) {
    const firstDay = new Date(year, month, 1).getDay();
    // Convertir dimanche (0) en 7, et ajuster pour commencer lundi
    return firstDay === 0 ? 6 : firstDay - 1;
  }

  // ====================================
  // Fonction : Obtenir le nombre de jours dans le mois
  // ====================================
  function getDaysInMonth(year, month) {
    return new Date(year, month + 1, 0).getDate();
  }

  // ====================================
  // Fonction : Formater une date en YYYY-MM-DD
  // ====================================
  function formatDate(date) {
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  }

  // ====================================
  // Fonction : Obtenir les événements pour une date
  // ====================================
  function getEventsForDate(dateStr) {
    return EVENTS.filter(event => event.date === dateStr)
      .sort((a, b) => a.time.localeCompare(b.time));
  }

  // ====================================
  // Fonction : Vérifier si une date a des événements
  // ====================================
  function hasEvents(dateStr) {
    return EVENTS.some(event => event.date === dateStr);
  }

  // ====================================
  // Fonction : Mettre à jour le titre du calendrier
  // ====================================
  function updateCalendarTitle() {
    const month = MONTH_NAMES[currentDate.getMonth()];
    const year = currentDate.getFullYear();
    calendarTitle.textContent = `${month} ${year}`;
  }

  // ====================================
  // Fonction : Générer le calendrier
  // ====================================
  function renderCalendar() {
    calendarDays.innerHTML = '';

    const year = currentDate.getFullYear();
    const month = currentDate.getMonth();

    const firstDay = getFirstDayOfMonth(year, month);
    const daysInMonth = getDaysInMonth(year, month);
    const daysInPrevMonth = getDaysInMonth(year, month - 1);

    // Jours du mois précédent
    for (let i = firstDay - 1; i >= 0; i--) {
      const day = daysInPrevMonth - i;
      const dayEl = createDayElement(day, true);
      calendarDays.appendChild(dayEl);
    }

    // Jours du mois actuel
    for (let day = 1; day <= daysInMonth; day++) {
      const date = new Date(year, month, day);
      date.setHours(0, 0, 0, 0);

      const dayEl = createDayElement(day, false, date);
      calendarDays.appendChild(dayEl);
    }

    // Jours du mois suivant (pour compléter la grille)
    const totalCells = calendarDays.children.length;
    const remainingCells = 42 - totalCells; // Grille de 6 semaines
    for (let day = 1; day <= remainingCells; day++) {
      const dayEl = createDayElement(day, true);
      calendarDays.appendChild(dayEl);
    }

    updateCalendarTitle();
  }

  // ====================================
  // Fonction : Créer un élément jour
  // ====================================
  function createDayElement(day, isOtherMonth, date = null) {
    const dayEl = document.createElement('div');
    dayEl.className = 'seedit-calendar-day';

    if (isOtherMonth) {
      dayEl.classList.add('seedit-calendar-day--other-month');
    }

    // Vérifier si c'est aujourd'hui
    if (date && date.getTime() === today.getTime()) {
      dayEl.classList.add('seedit-calendar-day--today');
    }

    // Vérifier si c'est le jour sélectionné
    if (selectedDate && date && date.getTime() === selectedDate.getTime()) {
      dayEl.classList.add('seedit-calendar-day--selected');
    }

    // Créer le contenu
    const numberEl = document.createElement('span');
    numberEl.className = 'seedit-calendar-day__number';
    numberEl.textContent = day;
    dayEl.appendChild(numberEl);

    // Ajouter un badge si le jour a des événements
    if (date && !isOtherMonth) {
      const dateStr = formatDate(date);
      const events = getEventsForDate(dateStr);

      if (events.length > 0) {
        const badgeEl = document.createElement('div');
        badgeEl.className = 'seedit-calendar-day__badge';

        // Afficher jusqu'à 3 points
        const dotCount = Math.min(events.length, 3);
        for (let i = 0; i < dotCount; i++) {
          const dotEl = document.createElement('span');
          dotEl.className = 'seedit-calendar-day__dot';
          badgeEl.appendChild(dotEl);
        }

        dayEl.appendChild(badgeEl);
      }

      // Événement de clic
      dayEl.addEventListener('click', () => selectDate(date));
      dayEl.setAttribute('tabindex', '0');
      dayEl.setAttribute('role', 'button');
      dayEl.setAttribute('aria-label', `Sélectionner le ${day} ${MONTH_NAMES[date.getMonth()]}`);

      // Accessibilité clavier
      dayEl.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          selectDate(date);
        }
      });
    }

    return dayEl;
  }

  // ====================================
  // Fonction : Sélectionner une date
  // ====================================
  function selectDate(date) {
    selectedDate = date;
    renderCalendar();
    showEventsForDate(date);
  }

  // ====================================
  // Fonction : Afficher les événements d'une date
  // ====================================
  function showEventsForDate(date) {
    const dateStr = formatDate(date);
    const events = getEventsForDate(dateStr);

    const day = date.getDate();
    const month = MONTH_NAMES[date.getMonth()];
    const year = date.getFullYear();

    eventsTitle.textContent = `Événements du ${day} ${month} ${year}`;

    if (events.length === 0) {
      eventsList.innerHTML = '<p class="seedit-calendar-widget__events-empty">Aucun événement ce jour-là</p>';
    } else {
      eventsList.innerHTML = '';
      events.forEach(event => {
        const eventEl = createEventElement(event);
        eventsList.appendChild(eventEl);
      });
    }
  }

  // ====================================
  // Fonction : Créer un élément événement
  // ====================================
  function createEventElement(event) {
    const eventEl = document.createElement('div');
    eventEl.className = `seedit-calendar-event seedit-calendar-event--${event.category}`;

    eventEl.innerHTML = `
      <div class="seedit-calendar-event__time">${event.time}</div>
      <div class="seedit-calendar-event__content">
        <div class="seedit-calendar-event__title">${escapeHtml(event.title)}</div>
        <div class="seedit-calendar-event__category">${CATEGORY_LABELS[event.category] || event.category}</div>
      </div>
    `;

    return eventEl;
  }

  // ====================================
  // Fonction : Échapper le HTML (sécurité XSS)
  // ====================================
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // ====================================
  // Fonction : Naviguer au mois précédent
  // ====================================
  function goToPrevMonth() {
    currentDate.setMonth(currentDate.getMonth() - 1);
    renderCalendar();
  }

  // ====================================
  // Fonction : Naviguer au mois suivant
  // ====================================
  function goToNextMonth() {
    currentDate.setMonth(currentDate.getMonth() + 1);
    renderCalendar();
  }

  // ====================================
  // Gestionnaires d'événements
  // ====================================
  prevMonthBtn.addEventListener('click', goToPrevMonth);
  nextMonthBtn.addEventListener('click', goToNextMonth);

  // Accessibilité clavier pour la navigation
  [prevMonthBtn, nextMonthBtn].forEach(btn => {
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        btn.click();
      }
    });
  });

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    renderCalendar();
  }

  // Lancer l'initialisation au chargement du DOM
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
