// ====================================
// Widget de Partage Social - Seed I.T
// ====================================

(function() {
  'use strict';

  // ====================================
  // Configuration
  // ====================================
  const SHARE_CONFIG = {
    // URL et texte par défaut (à personnaliser en production)
    url: window.location.href,
    title: 'Découvrez ce contenu incroyable !',
    description: 'Widget de partage social par Seed I.T - Partagez facilement sur vos réseaux préférés.',

    // Durée d'affichage du feedback (ms)
    feedbackDuration: 3000
  };

  // URLs de partage pour chaque plateforme
  const SHARE_URLS = {
    facebook: (url, title) => `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(url)}`,
    twitter: (url, title) => `https://twitter.com/intent/tweet?url=${encodeURIComponent(url)}&text=${encodeURIComponent(title)}`,
    linkedin: (url, title) => `https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(url)}`,
    whatsapp: (url, title) => `https://wa.me/?text=${encodeURIComponent(title + ' ' + url)}`,
    email: (url, title, description) => `mailto:?subject=${encodeURIComponent(title)}&body=${encodeURIComponent(description + '\n\n' + url)}`
  };

  // ====================================
  // Éléments du DOM
  // ====================================
  const widgets = document.querySelectorAll('.seedit-share-widget');

  // ====================================
  // Fonction : Ouvrir une fenêtre de partage
  // ====================================
  function openShareWindow(url, platform) {
    const width = 600;
    const height = 500;
    const left = (window.innerWidth - width) / 2;
    const top = (window.innerHeight - height) / 2;

    const features = `width=${width},height=${height},left=${left},top=${top},scrollbars=yes,resizable=yes`;

    window.open(url, `share-${platform}`, features);
  }

  // ====================================
  // Fonction : Copier le lien dans le presse-papiers
  // ====================================
  async function copyToClipboard(text, feedbackEl) {
    try {
      // Utiliser l'API Clipboard moderne
      if (navigator.clipboard && navigator.clipboard.writeText) {
        await navigator.clipboard.writeText(text);
        showFeedback(feedbackEl, true);
      } else {
        // Fallback pour les navigateurs anciens
        const textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();

        const success = document.execCommand('copy');
        document.body.removeChild(textarea);

        showFeedback(feedbackEl, success);
      }
    } catch (err) {
      showFeedback(feedbackEl, false);
    }
  }

  // ====================================
  // Fonction : Afficher le feedback de copie
  // ====================================
  function showFeedback(feedbackEl, success) {
    if (!feedbackEl) return;

    // Mettre à jour le message
    const messageEl = feedbackEl.querySelector('span');
    if (success) {
      messageEl.textContent = 'Lien copié !';
      feedbackEl.style.background = 'rgba(16, 172, 132, 0.15)';
      feedbackEl.style.borderColor = 'rgba(16, 172, 132, 0.3)';
      feedbackEl.style.color = '#10ac84';
    } else {
      messageEl.textContent = 'Erreur lors de la copie';
      feedbackEl.style.background = 'rgba(231, 76, 60, 0.15)';
      feedbackEl.style.borderColor = 'rgba(231, 76, 60, 0.3)';
      feedbackEl.style.color = '#e74c3c';
    }

    // Afficher le feedback
    feedbackEl.style.display = 'flex';

    // Masquer après un délai
    setTimeout(() => {
      feedbackEl.style.display = 'none';
    }, SHARE_CONFIG.feedbackDuration);
  }

  // ====================================
  // Fonction : Gérer le clic sur un bouton de partage
  // ====================================
  function handleShareClick(platform, widgetEl) {
    const url = SHARE_CONFIG.url;
    const title = SHARE_CONFIG.title;
    const description = SHARE_CONFIG.description;

    switch (platform) {
      case 'facebook':
        openShareWindow(SHARE_URLS.facebook(url, title), platform);
        break;

      case 'twitter':
        openShareWindow(SHARE_URLS.twitter(url, title), platform);
        break;

      case 'linkedin':
        openShareWindow(SHARE_URLS.linkedin(url, title), platform);
        break;

      case 'whatsapp':
        openShareWindow(SHARE_URLS.whatsapp(url, title), platform);
        break;

      case 'email':
        window.location.href = SHARE_URLS.email(url, title, description);
        break;

      case 'copy':
        const feedbackEl = widgetEl.querySelector('.seedit-share-widget__feedback');
        copyToClipboard(url, feedbackEl);
        break;

      default:
        break;
    }
  }

  // ====================================
  // Fonction : Initialiser un widget de partage
  // ====================================
  function initWidget(widgetEl) {
    const buttons = widgetEl.querySelectorAll('.seedit-share-btn');

    buttons.forEach(button => {
      button.addEventListener('click', (e) => {
        e.preventDefault();
        const platform = button.getAttribute('data-platform');
        if (platform) {
          handleShareClick(platform, widgetEl);
        }
      });

      // Support clavier (Enter et Space)
      button.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          const platform = button.getAttribute('data-platform');
          if (platform) {
            handleShareClick(platform, widgetEl);
          }
        }
      });
    });
  }

  // ====================================
  // Initialisation
  // ====================================
  function init() {
    widgets.forEach(widget => {
      initWidget(widget);
    });
  }

  // ====================================
  // Lancer l'initialisation au chargement du DOM
  // ====================================
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
